#include "BufferNSeconds.hpp"
#include "util/Random.hpp"

namespace twitch {
namespace analytics {
BufferNSeconds::BufferNSeconds(Listener& listener, std::shared_ptr<Scheduler> scheduler, MediaTime interval)
    : AnalyticsEvent(listener)
    , m_scheduler(std::move(scheduler))
    , m_interval(interval.milliseconds())
{
}

BufferNSeconds::~BufferNSeconds()
{
    m_outstanding.cancel();
}

void BufferNSeconds::onRebuffering(MediaTime startTime, const std::string& bufferSessionId, int bufferEmptyCount)
{
    int offsetMs = Random::integer(0, int(m_interval.count()) - 1);

    auto onUpdate = [this, startTime, bufferSessionId, bufferEmptyCount](int bufferOffset) {
        MediaTime cumulativeTime = MediaTime::now<Clock>() - startTime;
        json11::Json::object properties {
            { "buffer_time_offset", bufferOffset },
            { "seconds_buffered", cumulativeTime.seconds() },
            { "buffer_session_id", bufferSessionId },
            { "buffer_empty_count", bufferEmptyCount }
        };
        m_listener.onEventReady(*this, properties);
    };

    m_outstanding.cancel();
    m_outstanding = m_scheduler->schedule([this, onUpdate, offsetMs] {
        onUpdate(offsetMs);
        // 'buffer_time_offset' should be 0 after first event
        m_outstanding = m_scheduler->schedule(std::bind(onUpdate, 0), m_interval, true);
    },
        std::chrono::milliseconds(offsetMs));
}

void BufferNSeconds::onStateChanged(MediaTime timestamp, Player::State state)
{
    (void)timestamp;

    if (state != Player::State::Buffering) {
        m_outstanding.cancel();
    }
}

void BufferNSeconds::onPlayerSeek(MediaTime origin, MediaTime dest)
{
    (void)origin;
    (void)dest;
    m_outstanding.cancel();
}
}
}
