#include "MinuteWatched.hpp"
#include "player/MediaPlayer.hpp"
#include "util/Random.hpp"

namespace twitch {
namespace analytics {
MinuteWatched::MinuteWatched(Listener& listener, Player& player, std::shared_ptr<Scheduler> scheduler)
    : AnalyticsEvent(listener)
    , m_player(player)
    , m_state(Player::State::Idle)
    , m_scheduler(std::move(scheduler))
    , m_initialOffset(MediaTime::invalid())
    , m_minutesLogged(0)
    , m_isClip(false)
{
}

const char* MinuteWatched::getName() const
{
    return (m_isClip ? "clips_minute_watched" : "minute-watched");
}

MinuteWatched::~MinuteWatched()
{
    stop();
}

void MinuteWatched::pause()
{
    cancelTask();
    if (m_timerDuration >= MediaTime::zero()) {
        MediaTime elapsed = MediaTime::now() - m_timerStart;
        m_timerDuration = std::max(m_timerDuration - elapsed, MediaTime::zero());
    }
}

void MinuteWatched::start(MediaTime duration, MediaTime now)
{
    cancelTask();
    m_timerDuration = duration;
    resume(now);
}

void MinuteWatched::stop()
{
    cancelTask();
    m_timerDuration = MediaTime::zero();
}

void MinuteWatched::resume(MediaTime now)
{
    if (!m_task && m_timerDuration >= MediaTime::zero()) {
        m_timerStart = now;
        m_task = m_scheduler->schedule([this] {
            m_task.reset(); // mark that task is complete
            onTimerComplete();
        },
            m_timerDuration.microseconds());
    }
}

void MinuteWatched::cancelTask()
{
    if (m_task) {
        m_task->cancel();
        m_task.reset();
    }
}

void MinuteWatched::onError(const Error&)
{
    stop();
}

void MinuteWatched::onPlaySession(const PlaySession& session, MediaTime timestamp)
{
    (void)timestamp;
    m_isClip = session.isClip();

    // Reset to initial state
    stop();
    m_minutesLogged = 0;
    m_initialOffset = MediaTime::invalid();
    m_lastSegmentStartTime = MediaTime::zero();
    m_previousStatistics = m_player.getStatistics();
    if (isWatchableState()) {
        onStatePlay();
    }
}

void MinuteWatched::onStateChanged(MediaTime timestamp, Player::State state)
{
    (void)timestamp;
    m_state = state;

    if (isWatchableState()) {
        onStatePlay();
    } else {
        pause();
    }
}

void MinuteWatched::onStatePlay()
{
    MediaTime now = MediaTime::now();

    // Check if this is the first Playing state received for this video
    if (!m_initialOffset.valid()) {
        m_initialOffset = MediaTime(Random::real(0.0, 60.0));
        start(m_initialOffset, now);
    } else {
        resume(now);
    }
}

void MinuteWatched::onTimerComplete()
{
    // Fire minute watched event
    json11::Json::object properties;
    properties["time"] = MediaTime::now<Clock>().seconds();
    properties["seconds_offset"] = m_initialOffset.seconds();
    properties["minutes_logged"] = ++m_minutesLogged;
    properties["estimated_bandwidth"] = m_player.getBandwidthEstimate();

    if (m_player.getDuration() == MediaTime::max()) {
        properties["hls_latency_broadcaster"] = static_cast<int>(m_player.getLiveLatency().milliseconds().count());
    }

    const auto& statistics = m_player.getStatistics();
    properties["decoded_frames"] = statistics.getDecodedFrames() - m_previousStatistics.getDecodedFrames();
    properties["dropped_frames"] = statistics.getDroppedFrames() - m_previousStatistics.getDroppedFrames();
    int videoFramesRendered = statistics.getRenderedFrames() - m_previousStatistics.getRenderedFrames();
    properties["rendered_frames"] = videoFramesRendered;
    properties["current_fps"] = static_cast<float>(videoFramesRendered) / std::max(static_cast<float>(m_timerDuration.seconds()), 1.0f);
    properties["average_bitrate"] = m_player.getAverageBitrate();
    properties["playback_rate"] = m_player.getPlaybackRate();
    updateTransportHistory(properties);

    m_listener.onEventReady(*this, properties);

    // Store current stats for next update
    m_previousStatistics = statistics;

    if (isWatchableState()) {
        MediaTime now = MediaTime::now();
        MediaTime waitTime = (m_timerDuration - (now - m_timerStart)) + MediaTime(60.0);
        start(waitTime, now);
    }
}

void MinuteWatched::updateTransportHistory(json11::Json::object& properties)
{
    size_t bytes = 0;
    std::chrono::milliseconds downloadDuration(0);
    std::chrono::milliseconds segmentDuration(0);
    std::chrono::milliseconds firstByteLatency(0);
    int segments = 0;

    if (m_player.getName() == "mediaplayer") {
        MediaPlayer& mediaPlayer = static_cast<MediaPlayer&>(m_player);
        const auto& history = mediaPlayer.getQualitySelector().getTransferHistory();

        auto itr = std::find_if(history.begin(), history.end(), [this](const abr::RequestMetric transfer) {
            return transfer.initiated > m_lastSegmentStartTime;
        });

        for (; itr != history.end(); ++itr) {
            const auto& transfer = *itr;
            m_lastSegmentStartTime = transfer.initiated;
            bytes += transfer.bytes;
            downloadDuration += transfer.getDuration().milliseconds();
            firstByteLatency += transfer.getRequestLatency().milliseconds();
            segmentDuration += transfer.mediaDuration.milliseconds();
            segments++;
        }

        properties["transport_segments"] = segments;
        properties["transport_download_bytes"] = static_cast<int>(bytes);
        properties["transport_download_duration"] = static_cast<int>(downloadDuration.count());
        properties["transport_segment_duration"] = static_cast<int>(segmentDuration.count());
        properties["transport_first_byte_latency"] = static_cast<int>(firstByteLatency.count());
    }
}

bool MinuteWatched::isWatchableState() const
{
    bool hadInitialPlay = m_initialOffset.valid();
    return (m_state == Player::State::Playing) || ((m_state == Player::State::Buffering) && hadInitialPlay);
}
}
}
