#include "NSecondsWatched.hpp"
#include <cassert>

namespace twitch {
namespace analytics {
NSecondsWatched::NSecondsWatched(Listener& listener, std::shared_ptr<Scheduler> scheduler, std::vector<MediaTime>& watchTimes)
    : AnalyticsEvent(listener)
    , m_scheduler(std::move(scheduler))
    , m_watchTimes(std::move(watchTimes))
    , m_watchStarted(false)
{
}

NSecondsWatched::~NSecondsWatched()
{
    stop();
}

void NSecondsWatched::onError(const Error&)
{
    stop();
}

void NSecondsWatched::onStateChanged(MediaTime timestamp, Player::State state)
{
    (void)timestamp;

    if (state == Player::State::Playing) {
        if (!m_watchStarted) { // First watch state
            m_watchStarted = true;
            for (auto time : m_watchTimes) {
                m_tasks.push_back(start(time));
            }
        }
    } else if (state == Player::State::Idle || state == Player::State::Ended) {
        stop();
    }
}

void NSecondsWatched::onPlayerLoad(MediaTime timestamp, const std::string& path, bool keepPlaySession)
{
    (void)timestamp;
    (void)path;
    (void)keepPlaySession;
    m_watchStarted = false;
    m_tasks.clear();
}

void NSecondsWatched::onTimerComplete(MediaTime interval)
{
    json11::Json::object properties;

    properties["seconds_after_play"] = interval.seconds();
    m_listener.onEventReady(*this, properties);
}

std::shared_ptr<Cancellable> NSecondsWatched::start(MediaTime interval)
{
    assert(interval > MediaTime::zero());
    if (interval > MediaTime::zero()) {
        return m_scheduler->schedule(std::bind(&NSecondsWatched::onTimerComplete, this, interval),
            interval.milliseconds());
    }
    return nullptr;
}

void NSecondsWatched::stop()
{
    for (const auto& task : m_tasks) {
        task->cancel();
    }
    m_tasks.clear();
}
}
}
