#include "VideoError.hpp"
#include "MediaResult.hpp"

namespace twitch {
namespace analytics {
VideoError::VideoError(Listener& listener)
    : AnalyticsEvent(listener)
{
}

void VideoError::onError(const Error& error)
{
    // Source not available not reported since stream is generally offline
    if (error.result != MediaResult::ErrorNotAvailable || error.source != ErrorSource::Playlist) {
        populateError(error, false);
    }
}

void VideoError::onRecoverableError(const Error& error)
{
    // only decode/render errors are always reported
    if (error.source == ErrorSource::Decode || error.source == ErrorSource::Render) {
        populateError(error, true);
    }
}

void VideoError::populateError(const Error& error, bool recoverable)
{
    auto timestamp = MediaTime::now<std::chrono::system_clock>();

    json11::Json::object properties;
    properties["time"] = timestamp.seconds();
    properties["video_error_source"] = errorSourceString(error.source);
    properties["video_error_result"] = mediaResultString(error.result);
    properties["video_error_code"] = error.result.code;
    properties["video_error_value"] = error.result.value;
    properties["video_error_message"] = error.message;

    // flatten the usher error response since the json doesn't fit into the video-error table
    if (error.source == ErrorSource::Playlist) {
        std::string err;
        json11::Json json = json11::Json::parse(error.message, err);
        if (!json.array_items().empty()) {
            auto item = json.array_items()[0];
            properties["video_error_message"] = item["error"].string_value();
        }
    }

    properties["video_error_recoverable"] = recoverable;

    m_listener.onEventReady(*this, properties);
}
}
}
