#include "DownloadSource.hpp"
#include "FileStream.hpp"
#include "media/mp2t/Mp2TReader.hpp"
#include "media/mp4/Mp4Reader.hpp"
#include "playercore/platform/NativePlatform.hpp"

namespace twitch {
const std::chrono::seconds DefaultTimeoutSeconds(10);

DownloadSource::DownloadSource(MediaSource::Listener& listener,
    Platform& platform, const std::shared_ptr<Scheduler>& scheduler, const std::string& url)
    : ScopedScheduler(scheduler)
    , m_log(platform.getLog(), "Download ")
    , m_listener(listener)
    , m_readerListener(listener)
    , m_platform(platform)
    , m_httpClient(platform.createAsyncHttpClient(scheduler))
    , m_url(url)
    , m_readTimeout(DefaultTimeoutSeconds)
    , m_seekTime(MediaTime::zero())
    , m_request("File")
{
}

DownloadSource::~DownloadSource()
{
    close();
    cancel();
}

void DownloadSource::open()
{
    if (m_url.empty()) {
        m_listener.onSourceError(Error(ErrorSource::File, MediaResult::ErrorNotSupported, "Invalid url"));
        return;
    }

    if (!m_request.isComplete() && !m_request.isPending()) {
        downloadFile();
    }
}

void DownloadSource::close()
{
    cancelRequest();
}

void DownloadSource::seekTo(MediaTime time)
{
    m_seekTime = time;
    if (m_reader) {
        m_reader->seekTo(time);
    }
}

void DownloadSource::read(const TimeRange& range)
{
    if (m_reader) {
        if (m_readerListener.getTrackCount() > 0) {
            m_reader->readSamples(range.duration);
        } else if (m_request.isComplete()) {
            m_listener.onSourceError(Error(ErrorSource::File,
                MediaResult::ErrorInvalidData, "Request finished without parsing"));
        }
    }
}

MediaTime DownloadSource::getDuration() const
{
    return m_reader ? m_reader->getDuration() : MediaTime::zero();
}

void DownloadSource::downloadFile()
{
    m_log.debug("request %s", m_url.c_str());

    auto request = m_httpClient->createRequest(m_url, HttpMethod::GET);
    request->setTimeout(std::chrono::seconds(static_cast<int>(m_readTimeout.seconds())));
    m_request.onRequest(request);

    m_httpClient->send(
        request, [this](const std::shared_ptr<HttpResponse>& response) {
        m_request.onResponse(*response);
        m_log.info("http status %d", response->getStatus());
        if (response->isSuccess()) {
            response->setReadTimeout(std::chrono::seconds(static_cast<int>(m_readTimeout.seconds())));

            std::string contentType = response->getHeader("Content-Type");
            m_log.info("received content type %s", contentType.c_str());

            // check the content type
            if (!m_reader && !contentType.empty()) {
                MediaType type(contentType);
                // also accept application/octet-stream or binary/octet-stream as MP4s
                if (type.matches(MediaType::Video_MP4) || type.matches(MediaType::Audio_MP4)
                    || type.matches(MediaType::Application_OctetStream)
                    || type.matches(MediaType::Binary_OctetStream)) {

                    m_reader.reset(new media::Mp4Reader(m_platform, m_readerListener, m_url));
                } else if (type.matches(MediaType::Video_MP2T)) {
                    m_reader.reset(new media::Mp2TReader(m_platform, m_readerListener, m_url));
                }
            }

            if (!m_reader) {
                cancelRequest();
                m_listener.onSourceError(Error(ErrorSource::File, MediaResult::ErrorInvalidData,
                    "Invalid content type " + contentType));
                return;
            }

            using namespace std::placeholders;
            response->read(std::bind(&DownloadSource::onData, this, _1, _2, _3),
                    [=](int error) { onNetworkError("Response I/O error", error); });
        } else if (response->getStatus() == HttpStatusRequestedRangeNotSatisfiable) {
            // Range request not satisfiable, most likely the file is already complete
        } else {
            onNetworkError("Response HTTP error", response->getStatus());
        } },
        [=](int error) { onNetworkError("Request I/O error", error); });
}

void DownloadSource::onData(const uint8_t* data, size_t size, bool endOfStream)
{
    size_t offset = m_request.skipBytes(size);

    if (offset < size) {
        data += offset;
        size -= offset;
    } else if (size) {
        // skip over this data as it was already appended in the previous attempt
        return;
    }

    int initialTrackCount = m_readerListener.getTrackCount();
    m_reader->addData(data, size, endOfStream);
    m_request.appendedBytes(size, endOfStream);

    // check if enough data has been added to read the track info
    if (initialTrackCount == 0 && m_readerListener.getTrackCount() > 0) {
        m_reader->seekTo(m_seekTime);

        m_qualities.clear();

        auto format = m_reader->getTrackFormat(media::MediaReader::VideoTrackId);
        if (format) {
            m_quality.width = format->getInt(MediaFormat::Video_Width);
            m_quality.height = format->getInt(MediaFormat::Video_Height);
        }
        m_qualities.push_back(m_quality);

        m_listener.onSourceOpened();
        m_listener.onSourceDurationChanged(m_reader->getDuration());
    }

    if (m_readerListener.getTrackCount() > 0) {
        m_listener.onSourceFlush();
    }

    if (endOfStream && m_readerListener.getTrackCount() == 0) {
        m_listener.onSourceError(Error(ErrorSource::File, MediaResult::ErrorNotSupported, "Unsupported File"));
    }
}

void DownloadSource::onNetworkError(const std::string& message, int code)
{
    MediaResult result(MediaResult::ErrorNetwork, code);
    Error error(ErrorSource::File, result, message);
    if (m_request.isRetryable()) {
        m_listener.onSourceRecoverableError(error);
        m_request.retry(*this, [this]() { downloadFile(); });
    } else {
        m_listener.onSourceError(error);
    }
}

void DownloadSource::cancelRequest()
{
    m_request.cancel();
}
}
