#include "FileSource.hpp"
#include "media/mp2t/Mp2TReader.hpp"
#include "media/mp4/Mp4Reader.hpp"

namespace twitch {
FileSource::FileSource(MediaSource::Listener& listener, Platform& platform, const std::string& path)
    : m_listener(listener)
    , m_readerListener(listener)
    , m_platform(platform)
    , m_path(path)
{
    m_quality.name = path;
}

FileSource::~FileSource()
{
    close();
}

void FileSource::open()
{
    using namespace twitch::media;
    MediaType mediaType = MediaType::matchFromPath(m_path);

    if (mediaType.matches(MediaType::Video_MP2T)) {
        m_reader.reset(new Mp2TReader(m_platform, m_readerListener));
    } else if (mediaType.matches(MediaType::Video_MP4)) {
        m_reader.reset(new Mp4Reader(m_platform, m_readerListener));
    } else {
        m_listener.onSourceError(Error(ErrorSource::File, MediaResult::ErrorNotSupported, "Unsupported file type"));
        return;
    }

    m_reader->setStream(std::unique_ptr<Stream>(new FileStream(m_path, std::fstream::in)));

    m_qualities.clear();
    auto format = m_reader->getTrackFormat(MediaReader::VideoTrackId);
    if (format) {
        m_quality.width = format->getInt(MediaFormat::Video_Width);
        m_quality.height = format->getInt(MediaFormat::Video_Height);
    }
    m_qualities.push_back(m_quality);

    m_listener.onSourceOpened();
    m_listener.onSourceDurationChanged(m_reader->getDuration());
}

void FileSource::close()
{
    m_reader = nullptr;
}

void FileSource::seekTo(MediaTime time)
{
    if (m_reader) {
        m_reader->seekTo(time);
    }
}

void FileSource::read(const TimeRange& range)
{
    if (m_reader) {
        m_reader->readSamples(range.duration);
    }
}

MediaTime FileSource::getDuration() const
{
    return m_reader ? m_reader->getDuration() : MediaTime::zero();
}

const Quality& FileSource::getQuality() const
{
    return m_quality;
}
}
