#include "FileStream.hpp"
#include <cassert>
#include <cerrno>
#include <cstdio>

namespace twitch {
FileStream::FileStream(const std::string& path, std::fstream::openmode mode)
    : m_stream(path, mode)
    , m_length(0)
    , m_error(0)
{
    // Seek to the end of the file to update the length
    if (m_stream.good()) {
        m_stream.seekg(0, m_stream.end);
        int length = static_cast<int>(m_stream.tellg());
        if (m_stream.good()) {
            m_length = length;
            m_stream.seekg(0, m_stream.beg);
        }
    }
}

int64_t FileStream::read(uint8_t* buffer, size_t size)
{
    if (buffer == nullptr || m_error != 0 || position() < 0) {
        return -1;
    }

    if (size == 0) {
        return 0;
    }

    const size_t pos = static_cast<size_t>(position());
    size_t readSize = size;
    if (pos + readSize > m_length) {
        readSize = m_length - pos;
    }

    m_stream.read(reinterpret_cast<char*>(buffer), readSize);
    return error() != 0 ? -1 : readSize;
}

int64_t FileStream::write(const uint8_t* buffer, size_t size)
{
    if (buffer == nullptr && size > 0) {
        return -1;
    }

    if (m_error != 0) {
        return -1;
    }

    if (size == 0) {
        return 0;
    }

    m_stream.write(reinterpret_cast<const char*>(buffer), size);
    m_length += size;
    return error() != 0 ? -1 : size;
}

int64_t FileStream::length() const
{
    return m_error == 0 ? m_length : -1;
}

bool FileStream::seek(size_t position)
{
    if (m_error != 0 || position > m_length) {
        return false;
    }

    m_stream.seekg(std::streamoff(position), m_stream.beg);
    return error() == 0;
}

int64_t FileStream::position() const
{
    if (m_error != 0) {
        return -1;
    }
    auto pos = static_cast<int64_t>(m_stream.tellg());
    return error() != 0 ? -1 : pos;
}

bool FileStream::flush()
{
    if (m_error != 0) {
        return false;
    }
    m_stream.flush();
    return error() == 0;
}

void FileStream::close()
{
    if (m_stream.is_open()) {
        m_stream.close();
    }
}

int FileStream::error() const
{
    m_error = m_stream.good() ? 0 : errno;
    return m_error;
}
}
