#include "HlsSource.hpp"
#include "media/CodecString.hpp"
#include "media/MediaReaderFactory.hpp"
#include "media/SourceFormat.hpp"
#include "media/fourcc.hpp"
#include <algorithm>
#include <cstdlib>
#include <json11.hpp>

namespace twitch {
namespace hls {
HlsSource::HlsSource(MediaSource::Listener& listener,
    std::shared_ptr<Platform> platform,
    std::shared_ptr<Scheduler> scheduler,
    std::shared_ptr<HttpClient> httpClient,
    const std::string& url,
    Options options)
    : ScopedScheduler(std::move(scheduler))
    , m_listener(listener)
    , m_platform(platform)
    , m_httpClient(std::move(httpClient))
    , m_log(platform->getLog())
    , m_url(url)
    , m_options(options)
    , m_duration(MediaTime::zero())
    , m_masterRequest(MediaRequest::Type::MasterPlaylist)
    , m_readTimeout(MediaTime(10.0))
    , m_segmentPassthrough(false)
    , m_isAdaptive(false)
    , m_lowLatencyEnabled(true)
    , m_liveProgramStart(Segment::ProgramTimeNone)
{
}

HlsSource::~HlsSource()
{
    close();
    cancel();
}

void HlsSource::open()
{
    m_segmentRequests.clear();
    m_playlists.clear(); // reset any cached variants for live

    if (!m_master.parsed()) {
        loadMasterPlaylist(m_url);
    } else {
        m_listener.onSourceOpened();
    }

    // this track will always exist
    auto format = std::make_shared<media::SourceFormat>(MediaType::Text_Json);
    m_listener.onSourceTrack(media::MediaReader::MetaTrackId, format);
}

void HlsSource::close()
{
    cancelSegments();
    m_masterRequest.cancel();
    m_segmentRequests.clear();
    m_playlistRequests.clear();
    m_liveProgramStart = Segment::ProgramTimeNone;
    m_metadataQueue = std::queue<std::shared_ptr<MediaSampleBuffer>>();
}

void HlsSource::seekTo(MediaTime time)
{
    m_seekTime = time;
    for (auto& entry : m_segmentRequests) {
        entry.second.onSeek();
        updateSegmentSequence(entry.first, entry.second, time);
    }
}

void HlsSource::read(const TimeRange& range)
{
    m_readDuration = range.duration; // this is typically the target buffer size
    if (m_masterRequest.isFailed()) {
        return;
    }

    // determine renditions to check, currently handles muxed video, or demuxed video/audio
    // TODO handle subtitle/captions files
    std::vector<Rendition> renditions = { Rendition::Video };
    if (!m_streamInfo.audio.empty()) {
        renditions.push_back(Rendition::Audio);
    }

    for (Rendition rendition : renditions) {
        auto& request = m_segmentRequests[rendition];
        std::string playlistUrl = getPlaylistUrl(rendition);

        // check if we need the media playlist for the next segment
        if (request.isDiscontinuityAdaptive()) {
            m_streamInfo = getStream(m_selected);
            // reset the playlist url if the stream was changed
            playlistUrl = getPlaylistUrl(rendition);

            if (!m_playlists.count(playlistUrl)) {
                PlaylistUpdater& updater = m_playlistRequests[rendition];
                if (!updater.isPending() && updater.isRetryable()) {
                    updater.setScheduled(PlaylistUpdater::Clock::now());
                    loadMediaPlaylist(rendition);
                }
                return;
            }
        }

        const MediaPlaylist& playlist = m_playlists[playlistUrl];
        int sequenceNumber = request.getSequenceNumber();
        const auto& segments = playlist.segments();

        // update the sequence number for the next segment
        if (sequenceNumber < 0 || request.isDiscontinuitySeek()) {
            updateSegmentSequence(rendition, request, m_seekTime);
            sequenceNumber = request.getSequenceNumber();
        }

        if (m_isAdaptive && m_abrProbeRequest.isPending()) {
            // wait for the abr probe request to complete
            // TODO allow for parallel request
            break;
        }

        // create the request for downloading the next segment
        if (!request.isPending() && !segments.empty() && sequenceNumber != Segment::InvalidSequenceNumber) {
            // check live window
            int oldestSequenceNumber = segments.front().sequenceNumber;
            if (sequenceNumber < oldestSequenceNumber) {
                m_log->warn("Outside live window %d < %d", sequenceNumber, oldestSequenceNumber);
                m_listener.onSourceRecoverableError(
                    Error(ErrorSource::Playlist, MediaResult::Error, "Outside live window"));
                //updateSegmentSequence();
                // ideally would reset the sequence number, but it will be a discontinuity instead
                request.setSequenceNumber(oldestSequenceNumber);
            }

            for (const auto& segment : segments) {
                if (segment.sequenceNumber >= sequenceNumber) {
                    request.setSegment(segment);
                    if (request.isDiscontinuityInitialization(playlistUrl)
                        && !segment.getInitializationSegmentUrl(playlistUrl).empty()) {
                        // check if initialization segment is needed and download it first
                        request.setSegment(*segment.initializationSegment);
                    }
                    downloadSegment(request);
                    break;
                }
            }
        }
    }
}

bool HlsSource::isLive() const
{
    return m_duration == MediaTime::max();
}

void HlsSource::onMediaError(const Error& error)
{
    m_listener.onSourceError(error);
}

void HlsSource::onMediaFlush()
{
    m_listener.onSourceFlush();
}

void HlsSource::onMediaSample(int track, const std::shared_ptr<MediaSampleBuffer>& sample)
{
    // empty metadata queue aligned to the current sample time (for fmp4 time discontinuities)
    while (!m_metadataQueue.empty()) {
        auto meta = m_metadataQueue.front();
        m_metadataQueue.pop();
        meta->decodeTime = sample->decodeTime;
        meta->presentationTime = sample->presentationTime;
        m_listener.onSourceSample(media::MediaReader::MetaTrackId, meta);
    }
    m_listener.onSourceSample(track, sample);
}

void HlsSource::onMediaTrack(int track, const std::shared_ptr<MediaFormat>& format)
{
    format->setName(m_streamInfo.name);
    // always make sure the codec string is set for the web
    std::string parameters = getTrackCodecs(format->getType(), m_streamInfo.codecs);
    format->setType(MediaType(format->getType().name + parameters));
    m_listener.onSourceTrack(track, format);
}

bool HlsSource::isLowLatencySupported() const
{
    // checks if the master playlist if the stream has prefetch segment support
    const auto& properties = m_master.getSessionData();
    auto it = properties.find("FUTURE");
    bool supportsPrefetch = it != properties.end() && "true" == it->second;
    bool supportsABR = !m_isAdaptive || m_platform->getCapabilities().supportsLowLatencyABR;
    return m_lowLatencyEnabled && supportsPrefetch && supportsABR;
}

bool HlsSource::isSeekable() const
{
    // have to assume this to be seekable until any media playlist is loaded (if the player
    // seeks before playing the stream)
    return !isLive();
}

void HlsSource::setQuality(const Quality& quality, bool adaptive)
{
    m_isAdaptive = adaptive;
    for (auto& stream : m_master.getStreams()) {
        if (stream.name == quality.name) {
            m_log->info("Set quality to %s", quality.name.c_str());
            m_selected = quality;
            if (!adaptive) {
                cancelSegments();
            }
            // signal quality changing on all segments
            for (auto& entry : m_segmentRequests) {
                // reset all parsers if not adaptive
                entry.second.onVariantChange(adaptive);
            }
            // if live cancel remove any old variant playlist
            if (isLive()) {
                m_playlists.clear();
                for (auto& entry : m_playlistRequests) {
                    entry.second.cancel();
                }
            }
            break;
        }
    }
}

void HlsSource::loadMasterPlaylist(const std::string& url)
{
    if (url.empty()) {
        m_listener.onSourceError(
            Error(ErrorSource::Playlist, MediaResult::ErrorInvalidParameter, "Invalid HLS master playlist url"));
    } else {
        m_masterRequest.setUrl(url);
        m_masterRequest.setMaxAttempts(m_options.maxMasterPlaylistAttempts);
        downloadPlaylist(m_masterRequest, std::bind(&HlsSource::onMasterPlaylist, this, std::placeholders::_1));
    }
}

void HlsSource::onMasterPlaylist(const std::string& content)
{
    if (content.empty()) {
        m_listener.onSourceError(
            Error(ErrorSource::Playlist, MediaResult::ErrorInvalidData, "Empty HLS master playlist"));
        return;
    }

    std::vector<MasterPlaylist::StreamInformation> fallback;
    bool isMediaPlaylist = false;
    m_streamInfo = MasterPlaylist::StreamInformation();

    if (!m_master.parse(content, m_masterRequest.getUrl())) {
        // check if received a media playlist instead
        MasterPlaylist::StreamInformation info {};
        info.name = "unknown";
        info.video = "unknown";
        info.url = m_masterRequest.getUrl();
        MediaPlaylist playlist;
        playlist.parse(content, m_lowLatencyEnabled);
        isMediaPlaylist = playlist.parsed();
        if (isMediaPlaylist) { // this is actually a media playlist
            m_log->info("Received media playlist as master");
            fallback.push_back(info);
            m_streamInfo = info;
            m_playlists[info.url] = playlist;
        } else {
            MediaType type(m_masterRequest.getContentType());
            std::string message("Failed to parse HLS master playlist");
            if (!MediaType::Application_Apple_MPEG_URL.matches(type)
                && !MediaType::Application_MPEG_URL.matches(type)) {
                message += " invalid content type: " + m_masterRequest.getContentType();
            }
            m_listener.onSourceError(Error(ErrorSource::Playlist, MediaResult::ErrorInvalidData, message));
            return;
        }
    }

    // create a quality set from the stream information elements
    m_qualities.clear();
    for (auto& stream : isMediaPlaylist ? fallback : m_master.getStreams()) {
        Quality quality;
        quality.bitrate = stream.bandwidth;
        quality.framerate = stream.framerate;
        quality.width = stream.width;
        quality.height = stream.height;
        // this reformats the decimal codec avc string to hex
        quality.codecs = stream.codecs.format();

        // here we only look at the VIDEO group and base the name/group of that
        const MasterPlaylist::MediaInformation& video = m_master.getMedia(stream.video);
        quality.name = video.name.empty() ? stream.name : video.name;
        quality.group = video.group.empty() ? stream.video : video.group;
        quality.autoSelect = stream.video.empty() ? m_master.getMedia(stream.audio).autoSelect : video.autoSelect;
        // twitch specific hack to set the default quality
        if (video.group == "chunked") {
            quality.isDefault = true;
        }
        m_qualities.push_back(quality);
    }

    if (!m_qualities.empty()) {
        if (m_selected.name.empty()) {
            m_selected = m_qualities.front(); // default to first quality
        }
        setQuality(m_selected, false);
    }

    if (isMediaPlaylist) {
        PlaylistUpdater& updater = m_playlistRequests[Rendition::Video];
        updater.setScheduled(PlaylistUpdater::Clock::now());
        onMediaPlaylist(Rendition::Video, m_streamInfo.url, content, true);
    } else {
        // update Twitch specific playlist info
        m_listener.onSourceSessionData(m_master.getSessionData());
        // we need to load one variant in order to know the stream type
        loadMediaPlaylist(Rendition::Video, false);
    }

    m_listener.onSourceOpened();
}

void HlsSource::loadMediaPlaylist(Rendition rendition, bool poll)
{
    m_streamInfo = getStream(m_selected);
    std::string url = getPlaylistUrl(rendition);

    if (!url.empty()) {
        PlaylistUpdater& updater = m_playlistRequests[rendition];
        updater.setUrl(url);
        updater.setMaxAttempts(m_options.maxVariantPlaylistAttempts);
        downloadPlaylist(updater, std::bind(&HlsSource::onMediaPlaylist, this, rendition, url, std::placeholders::_1, poll));
    } else {
        m_log->warn("Empty variant url");
    }
}

void HlsSource::onMediaPlaylist(Rendition rendition, const std::string& url, const std::string& content, bool poll)
{
    MediaPlaylist& playlist = m_playlists[url];
    auto& request = m_segmentRequests[rendition];

    playlist.parse(content, isLowLatencySupported());

    MediaTime duration = playlist.isLive() ? MediaTime::max() : MediaTime(playlist.getDuration());

    if (!request.getReader()) {
        // check if this a fragmented MP4 playlist
        const MediaType& mediaType = playlist.getMediaType();
        media::MediaReader::TrackId track = media::MediaReader::AudioTrackId;
        switch (rendition) {
        case Rendition::Audio:
            track = media::MediaReader::AudioTrackId;
            break;
        case Rendition::Video:
            track = media::MediaReader::VideoTrackId;
            break;
        case Rendition::Subtitles:
        case Rendition::ClosedCaptions:
            track = media::MediaReader::MetaTrackId;
            break;
        }

        const auto& passthroughTypes = m_platform->getSupportedMediaTypes();
        // if the platform reports the type as directly supported the segments will not be parsed
        m_segmentPassthrough = passthroughTypes.find(mediaType) != passthroughTypes.end();

        // create reader for the segment type
        request.setRendition(rendition);
        request.setReader(m_platform->getMediaReaderFactory()->createReader(*m_platform, *this, mediaType, track, m_url));

        if (!request.getReader()) {
            m_listener.onSourceError(Error(ErrorSource::Source, MediaResult::ErrorNotSupported, "Unsupported segment type"));
            return;
        }

        preconfigureTracks(mediaType);
    }

    if (m_duration != duration && m_duration != MediaTime::max()) {
        m_duration = duration;
        m_listener.onSourceDurationChanged(m_duration);
        m_listener.onSourceSeekableChanged(isSeekable());
    }

    if (!request.isPending()) {
        m_listener.onSourceFlush(); // will trigger next segment download if not already in progress
    }

    if (playlist.isLive() || (playlist.isEvent() && !playlist.isEnded())) {
        if (!poll && !request.isPending()) {
            // unfortunately we can't keep this playlist, since it may become stale
            m_playlists.erase(url);
        } else {
            const auto& segments = playlist.segments();
            if (!segments.empty()) {
                if (playlist.isFinalSegment(request.getSequenceNumber())) {
                    m_listener.onSourceEndOfStream();
                } else {
                    PlaylistUpdater& updater = m_playlistRequests[rendition];
                    updater.schedulePlaylist(playlist, *this, [=]() { loadMediaPlaylist(rendition); });
                }
            }
        }
    } else if (playlist.segments().empty() || playlist.isFinalSegment(request.getSequenceNumber())) {
        m_listener.onSourceEndOfStream();
    }
}

void HlsSource::preconfigureTracks(const MediaType& streamType)
{
    // for the web configure the downstream sink in advance with the codec(s) string. Since MediaSource
    // SourceBuffers must be added once at initialization this is needed before appending any segment data
    if (m_platform->getName() == "web") {
        // Needed if loading a variant manifest directly
        using namespace twitch::media;
        CodecString codecString = m_streamInfo.codecs;
        if (codecString.values.empty() || m_master.getStreams().empty()) {
            codecString = CodecString::parse("mp4a.40.2,avc1.42001e");
        } else {
            auto streams = m_master.getStreams();
            // remove audio only streams
            streams.erase(std::remove_if(streams.begin(), streams.end(),
                              [this](const MasterPlaylist::StreamInformation& stream) { return isAudioOnly(stream.codecs); }),
                streams.end());
            // sort by bandwidth
            std::sort(streams.begin(), streams.end(),
                [](const MasterPlaylist::StreamInformation& a, const MasterPlaylist::StreamInformation& b) {
                    return a.bandwidth < b.bandwidth;
                });
            codecString = streams.front().codecs;
        }

        if (m_segmentPassthrough && m_streamInfo.audio.empty()) {
            // muxed content case or just 1 track
            auto format = std::make_shared<SourceFormat>(streamType);
            int track = streamType.isVideo() ? MediaReader::VideoTrackId : MediaReader::AudioTrackId;
            onMediaTrack(track, format);
        } else {
            // demuxed content case
            for (const auto& entry : codecString.values) {
                MediaType mediaType = CodecString::getMediaType(entry.first);
                if (!mediaType.name.empty()) {
                    int track = mediaType.isVideo() ? MediaReader::VideoTrackId : MediaReader::AudioTrackId;
                    auto format = std::make_shared<SourceFormat>(mediaType);
                    onMediaTrack(track, format);
                }
            }
        }
    }
}

std::string HlsSource::getTrackCodecs(const MediaType& type, const media::CodecString& codecs)
{
    using namespace twitch::media;
    if (m_segmentPassthrough && m_streamInfo.audio.empty()) {
        // muxed content case or just 1 track
        return ";codecs=\"" + codecs.format() + "\"";
    } else {
        // demuxed content case
        for (const auto& entry : codecs.values) {
            MediaType mediaType = CodecString::getMediaType(entry.first);
            if (mediaType.matches(type)) {
                return ";codecs=\"" + entry.first + "." + entry.second + "\"";
            }
        }
    }
    return "";
}

void HlsSource::cancelSegments()
{
    for (auto& entry : m_segmentRequests) {
        entry.second.cancel();
    }
    m_abrProbeRequest.cancel();
}

void HlsSource::downloadPlaylist(MediaRequest& request, const PlaylistHandler& handler)
{
    auto httpRequest = m_httpClient->createRequest(request.getUrl(), HttpMethod::GET);
    httpRequest->setTimeout(m_options.playlistTimeout);
    std::string accept;
    for (const auto& media : {
             MediaType::Application_MPEG_URL,
             MediaType::Application_Apple_MPEG_URL,
             MediaType::Application_Json,
             MediaType::Text_Plain }) {
        if (!accept.empty()) {
            accept += ", ";
        }
        accept += media.name;
    }
    httpRequest->setHeader("Accept", accept);

    request.setListener(&m_listener);
    request.onRequest(httpRequest);
    m_httpClient->send(
        httpRequest,
        [=, &request](std::shared_ptr<HttpResponse> response) { onPlaylistResponse(request, std::move(response), handler); },
        [=, &request](int error) { onPlaylistError(request, error, handler); });
}

void HlsSource::onPlaylistResponse(MediaRequest& request, const std::shared_ptr<HttpResponse>& response, PlaylistHandler handler)
{
    request.onResponse(*response);

    if (response->isSuccess()) {
        request.readString(
            *response, [&request, handler](const std::string& content) {
            request.onCompleted();
            handler(content); },
            [=, &request](int error) { onPlaylistError(request, error, handler); });
    } else {
        int status = response->getStatus();
        if (request.getType() == MediaRequest::Type::MasterPlaylist) {
            if (status == HttpStatusForbidden) {
                Error error(ErrorSource::Playlist, MediaResult(MediaResult::ErrorAuthorization, status),
                    "Failed to load playlist forbidden");
                request.readString(
                    *response, [=](const std::string& content) mutable {
                     if (!content.empty()) {
                         if (m_platform->getName() == "web") {
                             parseErrorContent(content, error);
                         } else {
                             error.message = content;
                         }
                     }
                     m_listener.onSourceError(error); },
                    [=](int) { m_listener.onSourceError(error); });
                return;
            } else if (status == HttpStatusBandwidthLimitExceeded
                || (status >= HttpStatusBadRequest
                    && status < HttpStatusInternalServerError
                    && status != HttpStatusTooManyRequests)) {
                Error error(ErrorSource::Playlist, MediaResult(MediaResult::ErrorNotAvailable, status),
                    "Failed to load playlist");
                m_listener.onSourceError(error);
                return;
            }
        }

        onPlaylistError(request, status, handler);
    }
}

void HlsSource::parseErrorContent(const std::string& content, Error& error)
{
    std::string err;
    json11::Json json = json11::Json::parse(content, err);
    if (!json.array_items().empty()) {
        auto item = json.array_items()[0];
        std::string contentErrorCode = item["error_code"].string_value();

        if (contentErrorCode == "content_geoblocked") {
            error.result.code = static_cast<int>(AuthorizationError::ErrorGeoblocked);
        } else if (contentErrorCode == "no_cdm_specified") {
            error.result.code = static_cast<int>(AuthorizationError::ErrorUnsupportedDevice);
        } else if (contentErrorCode == "anonymizer_blocked") {
            error.result.code = static_cast<int>(AuthorizationError::ErrorAnonymizerBlocked);
        } else if (contentErrorCode == "cellular_geoblocked") {
            error.result.code = static_cast<int>(AuthorizationError::ErrorCellularNetworkProhibited);
        } else if (contentErrorCode == "unauthorized_entitlements") {
            error.result.code = static_cast<int>(AuthorizationError::ErrorUnauthorizedEntitlements);
        } else if (contentErrorCode == "vod_manifest_restricted") {
            error.result.code = static_cast<int>(AuthorizationError::ErrorVodRestricted);
        }
        error.message = contentErrorCode;
    }
}

void HlsSource::onPlaylistError(MediaRequest& request, int error, const PlaylistHandler& handler)
{
    request.onNetworkError(error);
    Error ioError(ErrorSource::Playlist, MediaResult(MediaResult::ErrorNetwork, error), "Playlist error");

    if (!request.isRetryable()) {
        m_listener.onSourceError(ioError);
    } else {
        m_listener.onSourceRecoverableError(ioError);
        request.retry(*this, [=, &request]() { downloadPlaylist(request, handler); });
    }
}

void HlsSource::updateSegmentSequence(Rendition rendition, SegmentRequest& request, MediaTime time)
{
    std::string url = getPlaylistUrl(rendition);
    MediaPlaylist& playlist = m_playlists[url];
    const auto& segments = playlist.segments();

    if (segments.empty()) {
        if (!playlist.isEnded()) {
            m_log->info("No segments for stream %s", url.c_str());
            if (isLive()) {
                request.setSequenceNumber(Segment::InvalidSequenceNumber);
                loadMediaPlaylist(rendition);
            } else if (!m_playlistRequests[rendition].isPending()) {
                loadMediaPlaylist(rendition);
            }
        }
        return;
    }

    int sequenceNumber = Segment::InvalidSequenceNumber;
    // if live and not inside twitch specific preroll ad (when starting with a pre-roll ad time will be zero)
    if (isLive() && !isServerPreroll(segments.front())) {
        // check if a seek time was set for live/events
        if (m_seekTime != MediaTime::zero() && m_liveProgramStart != Segment::ProgramTimeNone) {
            const auto& segment = playlist.segmentAt(m_liveProgramStart + m_seekTime.milliseconds());
            if (segment.sequenceNumber != Segment::InvalidSequenceNumber) {
                sequenceNumber = segment.sequenceNumber;
                m_log->info("Live stream seek to sequence %d", sequenceNumber);
            }
        }
        // otherwise start from the end of the playlist based on the options
        if (sequenceNumber == Segment::InvalidSequenceNumber) {
            sequenceNumber = getLiveSequenceNumber(playlist);
        }
    } else if (playlist.isEvent() && !playlist.isEnded() && m_seekTime == MediaTime::zero()) {
        // TODO need a way to seek to live edge also with special time value, instead we assume
        // if time is zero we play from live
        sequenceNumber = getLiveSequenceNumber(playlist);
    } else {
        const auto& segment = playlist.segmentAt(time);
        sequenceNumber = segment.sequenceNumber;
    }

    if (sequenceNumber != request.getSequenceNumber()) {
        request.onSeek();
    }

    request.setSequenceNumber(sequenceNumber);
    m_log->info("Set %s sequence number to %d/%d",
        renditionToString(rendition), sequenceNumber, segments.back().sequenceNumber);
}

bool HlsSource::isServerPreroll(const Segment& segment) const
{
    for (const auto& dateRange : segment.dateRanges) {
        auto it = dateRange->attributes.find("CLASS");
        if (it != dateRange->attributes.end() && it->second == "twitch-stitched-ad") {
            return true;
        }
    }
    return false;
}

int HlsSource::getLiveSequenceNumber(const MediaPlaylist& playlist)
{
    // if the playlist specifies EXT-X-START tag use that segment
    int sequenceNumber = playlist.getStartSequence();
    // otherwise start from the back of the playlist base the number of live segments to use
    if (sequenceNumber == Segment::InvalidSequenceNumber) {
        const auto& segments = playlist.segments();
        MediaTime duration = MediaTime::zero();
        int liveSegments = playlist.getPrefetchSegmentCount() > 0
            ? (m_options.liveWindowSegments + playlist.getPrefetchSegmentCount() - m_options.prefetchOffset)
            : m_options.liveWindowSegments;
        for (auto it = segments.rbegin(); it != segments.rend(); ++it) {
            sequenceNumber = it->sequenceNumber;
            duration += it->duration;
            if (it - segments.rbegin() >= (liveSegments - 1) && duration >= m_readDuration) {
                break;
            }
        }
    }
    return sequenceNumber;
}

const MasterPlaylist::StreamInformation& HlsSource::getStream(const Quality& quality)
{
    for (const auto& stream : m_master.getStreams()) {
        if (stream.name == quality.name) {
            return stream;
        }
    }
    return m_streamInfo;
}

bool HlsSource::isAudioOnly(const media::CodecString& codecs)
{
    // checks if the codecs string contains only the mp4a codec id
    return codecs.values.size() == 1 && codecs.values.begin()->first == "mp4a";
}

std::string HlsSource::getPlaylistUrl(Rendition rendition)
{
    // using the given selected STREAM-INF get a playlist url for the target rendition type
    std::string url;
    switch (rendition) {
    case Rendition::Video:
        url = m_streamInfo.url.empty() ? m_master.getMedia(m_streamInfo.video).url : m_streamInfo.url;
        break;
    case Rendition::Audio:
        url = m_master.getMedia(m_streamInfo.audio).url;
        break;
    case Rendition::ClosedCaptions:
        url = m_master.getMedia(m_streamInfo.closedCaptions).url;
        break;
    case Rendition::Subtitles:
        url = m_master.getMedia(m_streamInfo.subtitles).url;
        break;
    }
    return url;
}

void HlsSource::downloadSegment(SegmentRequest& request)
{
    std::string playlistUrl = getPlaylistUrl(request.getRendition());
    const auto& segment = request.getSegment();
    logSegment(request.getRendition(), "Start", request.getSegment());

    // set the request properties
    request.setUrl(segment.getAbsoluteUrl(playlistUrl));
    request.setMaxAttempts(m_options.maxSegmentAttempts);
    request.setListener(segment.isInitialization ? nullptr : &m_listener);
    request.setMediaDuration(segment.duration);
    request.setMediaBitrate(m_streamInfo.bandwidth);

    auto httpRequest = m_httpClient->createRequest(request.getUrl(), HttpMethod::GET);
    httpRequest->setTimeout(std::chrono::seconds(static_cast<int>(m_readTimeout.seconds())));

    request.onRequest(httpRequest);
    m_httpClient->send(
        httpRequest,
        [=, &request](const std::shared_ptr<HttpResponse>& response) { onSegmentResponse(request, response); },
        [=, &request](int error) { onSegmentError(request, error); });
}

void HlsSource::onSegmentData(SegmentRequest& request, const uint8_t* data, size_t size, bool endOfStream)
{
    size_t offset = request.skipBytes(size);

    if (offset < size) {
        data += offset;
        size -= offset;
    } else if (size) {
        // skip over this data as it was already appended in the previous attempt
        return;
    }

    // add segment data
    if (request.getAppendedBytes() == 0) {
        media::MediaReader* reader = request.getReader();
        if (reader) {
            onSegmentDiscontinuity(request);
        }
    }
    // update request state
    request.addData(data, size, endOfStream);
    if (endOfStream) {
        onSegmentDownloaded(request);
    }
}

void HlsSource::onSegmentDiscontinuity(SegmentRequest& request)
{
    const auto& segment = request.getSegment();
    bool discontinuitySeek = request.isDiscontinuitySeek();
    media::MediaReader* reader = request.getReader();

    if (discontinuitySeek) {
        MediaTime seek;
        if (isLive()) {
            seek = m_seekTime;
        } else {
            seek = segment.cumulativeDuration - segment.duration;
        }
        m_log->info("Seek discontinuity to %lld us", seek.microseconds().count());
        reader->seekTo(seek);
    }

    request.onDiscontinuity();

    // generate segment metadata
    std::vector<std::shared_ptr<Segment::DateRange>> dateRanges = segment.dateRanges;
    if (discontinuitySeek) {
        // get metadata from previous segments applying to this segment
        const auto& playlist = m_playlists[getPlaylistUrl(request.getRendition())];
        dateRanges = playlist.getDateRanges(segment.sequenceNumber);
    }
    for (const auto& dateRange : dateRanges) {
        json11::Json attributes = json11::Json(dateRange->attributes);
        // TODO adjust the output timestamp based on the daterange
        double duration = dateRange->duration == Segment::DateRangeInfinite ? 0.0 : dateRange->duration;
        MediaTime timestamp = reader->getDuration();
        createMetadataSample(attributes.dump(), timestamp, MediaTime(duration), segment.discontinuity);
    }
}

void HlsSource::createMetadataSample(const std::string& content, MediaTime timestamp, MediaTime duration, bool discontinuity)
{
    using namespace twitch::media;
    auto sample = std::make_shared<MediaSampleBuffer>();
    sample->decodeTime = timestamp;
    sample->presentationTime = timestamp;
    sample->buffer = std::vector<uint8_t>(content.begin(), content.end());
    sample->duration = duration;
    sample->isDiscontinuity = discontinuity;
    sample->type = fourcc("json");
    m_metadataQueue.push(sample);
}

void HlsSource::onSegmentDownloaded(SegmentRequest& request)
{
    const auto& playlist = m_playlists[getPlaylistUrl(request.getRendition())];
    logSegment(request.getRendition(), "End", request.getSegment());

    // in low latency mode ABR make the probe request next
    size_t transferredBytes = request.getAppendedBytes();
    const int ABRProbeBytes = 32 * 1024;
    if (m_isAdaptive && !m_abrProbeRequest.isPending()
        && !request.isProbe()
        && request.isMediaPrefetch()
        && isLowLatencySupported() && transferredBytes > ABRProbeBytes) {
        auto segment = request.getSegment();
        segment.rangeOffset = 0;
        segment.rangeLength = ABRProbeBytes - 1;
        bool updateSegment = true;
        // only update the probe segment if it is no longer being served in the playlist
        for (const auto& s : playlist.segments()) {
            if (s.sequenceNumber == m_abrProbeRequest.getSegment().sequenceNumber) {
                updateSegment = false;
                break;
            }
        }
        if (updateSegment) {
            m_abrProbeRequest.setSegment(segment);
        }
        m_abrProbeRequest.setProbe(true);
        downloadSegment(m_abrProbeRequest);
    }

    // this increments the sequence number so log first
    request.onCompleted();

    if (playlist.isFinalSegment(request.getSegment().sequenceNumber)) {
        // for MP4 streams this forces the init fragment to be re-downloaded
        request.onVariantChange(false);
        m_listener.onSourceEndOfStream();
    }

    if (!request.getSegment().isInitialization) {
        // update the quality so changes are immediate in the next segment download
        m_listener.onSourceUpdateQuality();
    }
    m_listener.onSourceFlush();
}

void HlsSource::onSegmentError(SegmentRequest& request, int error)
{
    request.onNetworkError(error);

    if (!request.isProbe()) {
        Error ioError(ErrorSource::Segment, MediaResult(MediaResult::ErrorNetworkIO, error), "Segment download IO error");

        if (request.isRetryable()) {
            request.retry(*this, [=, &request]() { downloadSegment(request); });
            m_listener.onSourceRecoverableError(ioError);
        } else {
            cancelSegments();
            m_listener.onSourceError(ioError);
        }
    }
}

void HlsSource::onSegmentResponse(SegmentRequest& request, const std::shared_ptr<HttpResponse>& response)
{
    request.onResponse(*response);

    if (response->isSuccess()) {
        response->setReadTimeout(std::chrono::seconds(static_cast<int>(m_readTimeout.seconds())));
        response->read(
            [=, &request](const uint8_t* data, size_t size, bool endOfStream) {
                onSegmentData(request, data, size, endOfStream);
            },
            [=, &request](int error) { onSegmentError(request, error); });
    } else {
        int status = response->getStatus();
        Error httpError(ErrorSource::Segment, MediaResult(MediaResult::ErrorNetwork, status), "Segment download http error");

        if ((status < HttpStatusBadRequest || status >= HttpStatusInternalServerError) && request.isRetryable()) {
            request.retry(*this, [=, &request]() { downloadSegment(request); });
            m_listener.onSourceRecoverableError(httpError);
        } else {
            m_listener.onSourceError(httpError);
            cancelSegments();
        }
    }
}

void HlsSource::logSegment(Rendition rendition, const std::string& prefix, const Segment& segment)
{
    if (segment.isInitialization) {
        m_log->info(prefix + " initialization segment (%s) %s",
            m_streamInfo.name.c_str(),
            renditionToString(rendition));
    } else {
        const auto& playlist = m_playlists[getPlaylistUrl(rendition)];
        if (!playlist.segments().empty()) {
            m_log->info(prefix + " segment %d/%d stream %s (%s) %s",
                segment.sequenceNumber,
                playlist.segments().back().sequenceNumber,
                m_streamInfo.name.c_str(),
                renditionToString(rendition),
                segment.prefetch ? "prefetch" : "");
        }
    }
}
}
}
