#pragma once

#include "MasterPlaylist.hpp"
#include "PlaylistUpdater.hpp"
#include "Rendition.hpp"
#include "SegmentRequest.hpp"
#include "media/CodecString.hpp"
#include "media/MediaReader.hpp"
#include "player/MediaRequest.hpp"
#include "player/ScopedScheduler.hpp"
#include "playercore/MediaSource.hpp"
#include "playercore/platform/Platform.hpp"
#include <map>
#include <queue>

namespace twitch {
namespace hls {
class HlsSource : public MediaSource, private ScopedScheduler, private media::MediaReader::Listener {
public:
    /** HLS configurable options */
    struct Options {
        using Seconds = std::chrono::seconds;
        int liveWindowSegments = 3; /** Live min segments before the end to start playback */
        int prefetchOffset = 2; /** Number of prefech segments to use to startup logic */
        Seconds playlistTimeout = Seconds(30); /** HTTP playlist request timeout */
        int maxMasterPlaylistAttempts = 3; /** Maximum http request retry attempts */
        int maxVariantPlaylistAttempts = 5; /** Maximum http request retry attempts */
        int maxSegmentAttempts = 5; /** Maximum http request retry attempts */
        Options() {};
    };

    HlsSource(MediaSource::Listener& listener,
        std::shared_ptr<Platform> platform,
        std::shared_ptr<Scheduler> scheduler,
        std::shared_ptr<HttpClient> httpClient,
        const std::string& url,
        Options options = Options());
    ~HlsSource() override;
    HlsSource(const HlsSource&) = delete;
    const HlsSource& operator=(const HlsSource&) = delete;

    void open() override;
    void close() override;
    void seekTo(MediaTime time) override;
    void read(const TimeRange& range) override;
    bool isLive() const override;
    bool isPassthrough() const override { return false; }
    bool isSeekable() const override;
    MediaTime getDuration() const override { return m_duration; }
    const Quality& getQuality() const override { return m_selected; }
    void setQuality(const Quality& quality, bool adaptive) override;
    const std::vector<Quality>& getQualities() const override { return m_qualities; };
    void setReadTimeout(MediaTime time) override { m_readTimeout = time; }
    void setLowLatencyEnabled(bool enable) override { m_lowLatencyEnabled = enable; }

private:
    void onMediaDurationChanged(MediaTime) override {};
    void onMediaEndOfStream() override {};
    void onMediaError(const Error& error) override;
    void onMediaFlush() override;
    void onMediaSample(int track, const std::shared_ptr<MediaSampleBuffer>& sample) override;
    void onMediaTrack(int track, const std::shared_ptr<MediaFormat>& format) override;

    bool isLowLatencySupported() const;
    bool isServerPreroll(const Segment& segment) const;
    void loadMasterPlaylist(const std::string& url);
    void loadMediaPlaylist(Rendition rendition, bool poll = true);
    void onMasterPlaylist(const std::string& content);
    void onMediaPlaylist(Rendition rendition, const std::string& url, const std::string& content, bool poll);
    void preconfigureTracks(const MediaType& mediaType);
    void cancelSegments();
    void updateSegmentSequence(Rendition rendition, SegmentRequest& request, MediaTime time = MediaTime::zero());
    int getLiveSequenceNumber(const MediaPlaylist& playlist);
    std::string getTrackCodecs(const MediaType& type, const media::CodecString& codecs);
    using PlaylistHandler = std::function<void(const std::string& playlist)>;
    void downloadPlaylist(MediaRequest& request, const PlaylistHandler& handler);
    void onPlaylistResponse(MediaRequest& request, const std::shared_ptr<HttpResponse>& response, PlaylistHandler handler);
    void onPlaylistError(MediaRequest& request, int error, const PlaylistHandler& handler);
    void downloadSegment(SegmentRequest& request);
    const MasterPlaylist::StreamInformation& getStream(const Quality& quality);
    bool isAudioOnly(const media::CodecString& codecs);
    std::string getPlaylistUrl(Rendition rendition);

    void onSegmentData(SegmentRequest& request, const uint8_t* data, size_t size, bool endOfStream);
    void onSegmentDiscontinuity(SegmentRequest& request);
    void onSegmentDownloaded(SegmentRequest& request);
    void onSegmentError(SegmentRequest& request, int error);
    void onSegmentResponse(SegmentRequest& request, const std::shared_ptr<HttpResponse>& response);
    void createMetadataSample(const std::string& content, MediaTime timestamp, MediaTime duration, bool discontinuity);
    void logSegment(Rendition rendition, const std::string& prefix, const Segment& segment);
    void parseErrorContent(const std::string& content, Error& error);

    MediaSource::Listener& m_listener;
    std::shared_ptr<Platform> m_platform;
    std::shared_ptr<HttpClient> m_httpClient;
    std::shared_ptr<Log> m_log;
    const std::string m_url;
    const Options m_options;
    std::vector<Quality> m_qualities;
    Quality m_selected;
    MediaTime m_duration;
    MasterPlaylist m_master;
    MasterPlaylist::StreamInformation m_streamInfo;
    std::map<std::string, MediaPlaylist> m_playlists;
    std::map<Rendition, SegmentRequest> m_segmentRequests;
    std::map<Rendition, PlaylistUpdater> m_playlistRequests;
    SegmentRequest m_abrProbeRequest;
    MediaRequest m_masterRequest;
    MediaTime m_seekTime;
    MediaTime m_readDuration;
    MediaTime m_readTimeout;
    bool m_segmentPassthrough;
    bool m_isAdaptive;
    bool m_lowLatencyEnabled;
    Segment::ProgramTime m_liveProgramStart;
    std::queue<std::shared_ptr<MediaSampleBuffer>> m_metadataQueue;
};
}
}
