#include "MasterPlaylist.hpp"
#include "debug/trace.hpp"
#include <cstdlib>

namespace twitch {
namespace hls {
const static std::string EXT_X_STREAM_INF = "EXT-X-STREAM-INF:";
const static std::string EXT_X_I_FRAME_STREAM_INF = "EXT-X-I-FRAME-STREAM-INF:";
const static std::string EXT_X_MEDIA = "EXT-X-MEDIA:";
const static std::string EXT_X_TWITCH_INFO = "EXT-X-TWITCH-INFO:";
const static std::string EXT_X_SESSION_DATA = "EXT-X-SESSION-DATA:";
const static std::string EXT_X_INDEPENDENT_SEGMENTS = "EXT-X-INDEPENDENT-SEGMENTS";

MasterPlaylist::MasterPlaylist()
    : m_independentSegments(false)
{
}

bool MasterPlaylist::parse(const std::string& content, const std::string& baseURL)
{
    PlaylistParser parser(content);
    parser.nextLine();

    if (!parser.readM3U()) {
        return false;
    }

    while (parser.nextLine()) {
        if (!parser.readCommentStart()) {
            continue;
        }

        if (parser.readPrefix(EXT_X_MEDIA)) {

            std::map<std::string, std::string> attributes;
            if (!parser.parseAttributes(attributes)) {
                TRACE_WARN("Error parsing attributes: %s", parser.getLine().c_str());
                continue;
            }

            const std::string& group = attributes["GROUP-ID"];
            MediaInformation& media = m_groups[group];
            media.group = group;
            media.type = attributes["TYPE"];
            media.name = attributes["NAME"];
            media.language = attributes["LANGUAGE"];
            media.isDefault = attributes["DEFAULT"] == "YES";
            media.autoSelect = attributes["AUTOSELECT"] == "YES";

            const auto& uri = attributes["URI"];
            if (!uri.empty()) {
                media.url = getAbsoluteURL(baseURL, uri);
            }

        } else if (parser.readPrefix(EXT_X_STREAM_INF)) {

            m_streams.emplace_back();
            StreamInformation& stream = m_streams.back();
            readStream(parser, stream, baseURL);

        } else if (parser.readPrefix(EXT_X_I_FRAME_STREAM_INF)) {

            m_iFrameStreams.emplace_back();
            StreamInformation& stream = m_iFrameStreams.back();
            readStream(parser, stream, baseURL);

        } else if (parser.readPrefix(EXT_X_TWITCH_INFO)) {

            if (!parser.parseAttributes(m_sessionData)) {
                TRACE_WARN("Error parsing attributes: %s", parser.getLine().c_str());
            }
        } else if (parser.readPrefix(EXT_X_SESSION_DATA)) {

            if (!parser.parseAttributes(m_sessionData)) {
                TRACE_WARN("Error parsing attributes: %s", parser.getLine().c_str());
            }
        } else if (parser.readPrefix(EXT_X_INDEPENDENT_SEGMENTS)) {
            m_independentSegments = true;
        }
    }

    return parsed();
}

void MasterPlaylist::readStream(PlaylistParser& parser, StreamInformation& stream, const std::string& baseUrl)
{
    std::map<std::string, std::string> attributes;
    if (!parser.parseAttributes(attributes)) {
        TRACE_WARN("Error parsing attributes: %s", parser.getLine().c_str());
        return;
    };

    stream.codecs = media::CodecString::parse(attributes["CODECS"]);
    stream.video = attributes["VIDEO"];
    stream.audio = attributes["AUDIO"];
    stream.closedCaptions = attributes["CLOSED-CAPTIONS"];
    stream.subtitles = attributes["SUBTITLES"];
    stream.bandwidth = parser.parseInt(attributes["BANDWIDTH"]);

    const std::string& resolution = attributes["RESOLUTION"];
    auto position = !resolution.empty() ? resolution.find('x') : std::string::npos;

    if (position != std::string::npos) {
        stream.width = parser.parseInt(resolution.substr(0, position));
        stream.height = parser.parseInt(resolution.substr(position + 1, resolution.length()));
    } else {
        stream.width = 0;
        stream.height = 0;
    }

    const std::string& frameRate = attributes["FRAME-RATE"];
    if (!frameRate.empty()) {
        stream.framerate = std::strtof(frameRate.c_str(), nullptr);
    }

    const auto& uri = attributes["URI"];

    if (uri.empty()) {
        // The URL for the sequence is the next line.
        parser.nextLine();
        stream.url = getAbsoluteURL(baseUrl, parser.getLine());
    } else {
        stream.url = getAbsoluteURL(baseUrl, uri);
    }

    // Assign the VIDEO name to the stream for compatibility
    stream.name = getMedia(stream.video).name;
    // fallback name use the resolution
    if (stream.name.empty() && !resolution.empty()) {
        stream.name = resolution;
    }
    // fallback use the bandwidth
    if (stream.name.empty()) {
        stream.name = std::to_string(stream.bandwidth / 1000) + "kbps";
    }
}

bool MasterPlaylist::parsed() const
{
    return !m_streams.empty();
}

const std::vector<MasterPlaylist::StreamInformation>& MasterPlaylist::getStreams() const
{
    return m_streams;
}

const std::vector<MasterPlaylist::StreamInformation>& MasterPlaylist::getIFrameStreams() const
{
    return m_iFrameStreams;
}

const MasterPlaylist::MediaInformation& MasterPlaylist::getMedia(const std::string& group) const
{
    static MasterPlaylist::MediaInformation Empty;
    auto it = m_groups.find(group);
    return it != m_groups.end() ? it->second : Empty;
}

const std::map<std::string, std::string>& MasterPlaylist::getSessionData() const
{
    return m_sessionData;
}
}
}
