#pragma once

#include "PlaylistParser.hpp"
#include "media/CodecString.hpp"
#include <map>
#include <string>
#include <vector>

namespace twitch {
namespace hls {
/** Represents the content of a HLS master playlist */
class MasterPlaylist final {
public:
    MasterPlaylist();
    ~MasterPlaylist() = default;

    /** Represents the content of a EXT-X-STREAM-INF entry */
    struct StreamInformation {
        std::string name; // for compatibility
        std::string url;
        int width = 0;
        int height = 0;
        int bandwidth = 0;
        float framerate = 0;
        media::CodecString codecs;
        std::string video;
        std::string audio;
        std::string subtitles;
        std::string closedCaptions;
    };

    /** Represents the content of a EXT-X-MEDIA entry */
    struct MediaInformation {
        std::string name;
        std::string url;
        std::string group;
        std::string language;
        std::string type;
        bool isDefault = false;
        bool autoSelect = false;
    };

    /**
     * Parse playlist file that has been buffered into the string 'playlist'
     * Relative URLs are resolved relative to 'baseURL'. All other functions
     * on 'MasterPlaylist' return empty values before this is called.
     * @param playlist content
     * @param baseUrl base playlist url
     * @return true if the parse was successful
     */
    bool parse(const std::string& playlist, const std::string& baseUrl);

    /** @return true if the playlist was parsed, false otherwise */
    bool parsed() const;

    /**
     * @return Get the available streams corresponding to EXT-X-STREAM-INF entries
     */
    const std::vector<StreamInformation>& getStreams() const;

    /**
     * @return Get the I-FRAME only streams corresponding to EXT-X-I-FRAME-STREAM-INF entries
     */
    const std::vector<StreamInformation>& getIFrameStreams() const;

    /**
     * Get media by it's GROUP-ID
     * @param group
     * @return Returns the media info for the group or empty if not present
     */
    const MediaInformation& getMedia(const std::string& group) const;

    /** @return Get an extra info specified in the manifest */
    const std::map<std::string, std::string>& getSessionData() const;

    /** @return True if the segments are guaranteed to be keyframe aligned */
    bool isIndependentSegments() const { return m_independentSegments; }

private:
    void readStream(PlaylistParser& parser, StreamInformation& stream, const std::string& baseUrl);

    std::vector<StreamInformation> m_streams;
    std::vector<StreamInformation> m_iFrameStreams;
    std::map<std::string, MediaInformation> m_groups;
    std::map<std::string, std::string> m_sessionData;
    bool m_independentSegments;
};
}
}
