#pragma once

#include "Segment.hpp"
#include "playercore/MediaTime.hpp"
#include "playercore/MediaType.hpp"
#include <chrono>
#include <limits>
#include <map>
#include <memory>
#include <string>
#include <vector>

namespace twitch {
namespace hls {
/** Represents the content of a HLS media playlist */
class MediaPlaylist final {
public:
    // Possible playlist types
    static const std::string TypeEvent; //"EVENT"
    static const std::string TypeVOD; //"VOD"

    MediaPlaylist();
    ~MediaPlaylist() = default;

    /**
     * parse a playlist and return new segments not found in a previous parse (returns segments all
     * on first call)
     *
     * @param playlist
     * @param prefetch true to parse prefetch segments, false otherwise
     */
    void parse(const std::string& playlist, bool prefetch = true);

    /** Generate a playlist */
    std::string generate();

    /** @return true if the playlist was parsed, false otherwise */
    bool parsed() const;

    /** @return the playlist type, one of "EVENT", "VOD" */
    const std::string& getType() const;

    bool isEnded() const;
    MediaTime getDuration() const;
    int getStartSequence() const;
    int getTargetDuration() const;

    const Segment& segmentAt(MediaTime time) const;
    const Segment& segmentAt(Segment::ProgramTime datetime) const;
    bool isEvent() const { return m_type == TypeEvent; }
    bool isVOD() const { return m_type == TypeVOD; }
    bool isLive() const { return m_type.empty(); }
    bool isFinalSegment(int sequenceNumber) const;
    bool isIndependentSegments() const { return m_independentSegments; }
    const std::vector<Segment>& segments() const;
    int getPrefetchSegmentCount() const { return m_prefetchSegmentCount; }
    const MediaType& getMediaType() const { return m_mediaType; }
    std::vector<std::shared_ptr<Segment::DateRange>> getDateRanges(int sequenceNumber) const;

    static std::string generateIso8601(Segment::ProgramTime datetime);
    std::vector<Segment>& segments() { return m_segments; }

    void setVersion(int version) { m_version = version; }
    void setTargetDuration(int duration) { m_targetDuration = duration; }
    void setEndOfStream(bool ended) { m_ended = ended; }

private:
    int m_version;
    std::string m_type;
    MediaType m_mediaType;
    std::vector<Segment> m_segments;
    int m_targetDuration;
    bool m_ended;
    bool m_independentSegments;
    int m_prefetchSegmentCount;
    MediaTime m_twitchElapsedTime;
};
}
}
