#include "PlaylistParser.hpp"
#include <cstdio>
#include <cstdlib>
#include <ctime>

namespace twitch {
namespace hls {
PlaylistParser::PlaylistParser(const std::string& content)
    : m_buffer(content)
    , m_offset(0)
{
}

// Read the current line into 'line'. Return false if we're at EOF
bool PlaylistParser::nextLine()
{
    if (m_offset >= m_buffer.size()) {
        return false;
    }

    m_line.clear();
    size_t start = m_offset;
    size_t end = start;

    const char CR = '\r';
    const char LF = '\n';

    while (end < m_buffer.size() && m_buffer[end] != LF) {
        end++;
    }

    if (end > 0 && m_buffer[end - 1] == CR) {
        m_line.append(m_buffer, m_offset, (end - 1) - start);
    } else {
        m_line.append(m_buffer, m_offset, end - start);
    }

    m_offset = end + 1;
    return true;
}

int PlaylistParser::parseInt()
{
    return parseInt(m_line);
}

int PlaylistParser::parseInt(const std::string& str)
{
    return static_cast<int>(std::strtod(str.c_str(), nullptr));
}

double PlaylistParser::parseDouble()
{
    return std::strtod(m_line.c_str(), nullptr);
}

bool PlaylistParser::hasPrefix(const std::string& prefix)
{
    return m_line.compare(0, prefix.size(), prefix) == 0;
}

bool PlaylistParser::readPrefix(const std::string& prefix)
{
    if (hasPrefix(prefix)) {
        m_line.erase(0, prefix.size());
        return true;
    }

    return false;
}

bool PlaylistParser::readM3U()
{
    return readPrefix("#EXTM3U");
}

bool PlaylistParser::readCommentStart()
{
    return readPrefix("#");
}

// parse a comma separated list of key=value. Returns false on parse error
bool PlaylistParser::parseAttributes(std::map<std::string, std::string>& attributes)
{
    size_t end;
    bool quoted;
    size_t size = m_line.size();
    // skip for optional `:` at start of string
    size_t start = m_line.find_first_not_of(':');

    while (start < size) {
        // Find the equals sign between the key and value.
        end = m_line.find('=', start);

        // No equal found, but we still have line to parse.
        if (end == std::string::npos) {
            return false;
        }

        // Key has zero size, something is wrong.
        if (end == start) {
            return false;
        }

        std::string key = m_line.substr(start, end - start);

        // Skip over the = sign.
        start = end + 1;

        // Value is out of bounds, something is wrong.
        if (start > size) {
            return false;
        }

        // Values can be double quoted.
        quoted = m_line[start] == '"';

        if (quoted) {
            // Skip the quote.
            start += 1;

            // Find the matching end quote.
            end = m_line.find('"', start);

            // No matching end quote.
            if (end == std::string::npos) {
                return false;
            }
        } else {
            // Find the comma between the next pair.
            end = m_line.find(',', start);

            // No comma found, value must end the line.
            if (end == std::string::npos) {
                end = size;
            }
        }

        std::string value = m_line.substr(start, end - start);

        // Skip the comma.
        start = end + 1;

        // Also skip the trailing quote.
        if (quoted) {
            start += 1;
        }
        // Save the result in the given AttributeList.
        attributes[key] = value;
    }
    return true;
}

std::chrono::system_clock::time_point PlaylistParser::parseIso8601(const std::string& date)
{
    // parses dates in the format 2017-06-22T00:19:35Z
    std::tm time {};
    float seconds;
    sscanf(date.c_str(), "%d-%d-%dT%d:%d:%fZ",
        &time.tm_year, &time.tm_mon, &time.tm_mday, &time.tm_hour, &time.tm_min, &seconds);
    time.tm_year -= 1900; // year is since 1900
    time.tm_mon -= 1; // value is from 0-11
    time.tm_sec = static_cast<int>(seconds);
    time_t timet = std::mktime(&time);

    // non ideal way to get the timezone offset...
    time_t now = std::time(nullptr);
    std::tm* tm {};
    tm = std::gmtime(&now);
    time_t utc = std::mktime(tm);
    tm = localtime(&now);
    // TODO DST? check tm->tm_isdst
    int offset = static_cast<int>(now - utc);
    timet += offset;

    return std::chrono::system_clock::from_time_t(timet);
}
}
}
