#pragma once

#include <chrono>
#include <map>
#include <string>

namespace twitch {
namespace hls {
/** Takes a string buffer for a text file and provides readline and parsing functionality */
class PlaylistParser {
public:
    /**
     * @param content stored as a reference so must have a lifetime greater than this class.
     */
    explicit PlaylistParser(const std::string& content);

    /**
     * Read current line into a string.
     * @return Returns false for EOF
     */
    bool nextLine();

    /**
     * Reads the remaining part of the line as an integer.
     * @return int value
     */
    int parseInt();

    /**
     * Parses the given string as an integer
     * @param str string to parse
     * @return int value
     */
    int parseInt(const std::string& str);

    /**
     * Reads the remaining part of the line as an double.
     * @return double value
     */
    double parseDouble();

    /**
     * @param prefix to check
     * @return true if the line starts with the prefix, false otherwise
     */
    bool hasPrefix(const std::string& prefix);

    /**
     * Check if 'line' begins with 'prefix', if it does, consume 'prefix' from 'line'
     * @param prefix to check
     * @return true if consumed, false otherwise
     */
    bool readPrefix(const std::string& prefix);

    /** Read the #EXTM3U start of playlist tag */
    bool readM3U();

    /** Read the comment start # character */
    bool readCommentStart();

    /**
     * fill map from "key=value" pairs in "line"
     * @param attributes output attributes
     * @return false if the attributes couldn't be parsed
     */
    bool parseAttributes(std::map<std::string, std::string>& attributes);

    /**
     * Parses an ISO 8601 date from the playlist
     * @param date ISO 8601 input string
     * @return the data as a system clock time point
     */
    std::chrono::system_clock::time_point parseIso8601(const std::string& date);

    /**
     * @return current line
     */
    const std::string& getLine() { return m_line; }

private:
    const std::string& m_buffer;
    std::string m_line;
    size_t m_offset;
};

// Return absolute path from base and relative path
std::string getAbsoluteURL(const std::string& base, const std::string& relative);
}
}
