#include "PlaylistUpdater.hpp"
#include "util/Random.hpp"

namespace twitch {
namespace hls {
using std::chrono::milliseconds;

PlaylistUpdater::PlaylistUpdater()
    : MediaRequest(MediaRequest::Type::MediaPlaylist)
    , m_scheduled(TimePoint::min())
    , m_latestSequenceNumber(Segment::InvalidSequenceNumber)
    , m_staleVariantCount(0)
{
}

PlaylistUpdater::~PlaylistUpdater()
{
    cancel();
}

MediaTime meanDuration(const std::vector<Segment>& segments)
{
    MediaTime duration(0);
    int segmentCount(0);
    for (const auto& segment : segments) {
        if (segment.duration > MediaTime::zero()) {
            duration += segment.duration;
            segmentCount++;
        }
    }
    return (segmentCount == 0) ? duration : duration / segmentCount;
}

PlaylistUpdater::TimePoint PlaylistUpdater::next(const MediaPlaylist& playlist)
{
    const auto& segments = playlist.segments();
    if (segments.empty()) {
        return TimePoint::min();
    }

    int latestSequenceNumber = segments.back().sequenceNumber;

    const milliseconds minDelay(1000);
    const milliseconds amortizedDelay(250);

    if (m_scheduled == TimePoint::min()) {
        m_scheduled = Clock::now();
    }

    TimePoint when;

    if (latestSequenceNumber == m_latestSequenceNumber && playlist.isLive()) {
        // more aggressively check for new segments on live streams (non event ones)
        m_staleVariantCount++;
        when = m_scheduled + Random::jitter(milliseconds(100), minDelay);
        //m_log->info("Updated playlist with no new segments available");
    } else {
        const milliseconds epsilon(-5);
        // Poll at amortized rate by shortening the next delay
        auto mean = meanDuration(segments);
        auto waitTime = mean.milliseconds() + epsilon - m_staleVariantCount * (minDelay - amortizedDelay);
        when = m_scheduled + std::max(minDelay, waitTime);
        m_staleVariantCount = 0;
    }

    if (m_latestSequenceNumber != Segment::InvalidSequenceNumber
        && latestSequenceNumber - m_latestSequenceNumber > 1) {
        //m_log->info("Updated playlist with %d segments", latestSequenceNumber - m_latestSequenceNumber);
    }

    m_latestSequenceNumber = latestSequenceNumber;

    return when;
}

void PlaylistUpdater::reset()
{
    MediaRequest::cancel();
    m_scheduled = TimePoint::min();
    m_latestSequenceNumber = Segment::InvalidSequenceNumber;
}

void PlaylistUpdater::schedulePlaylist(const MediaPlaylist& playlist, Scheduler& scheduler, Scheduler::Action action)
{
    auto now = PlaylistUpdater::Clock::now();
    auto when = next(playlist);
    setScheduled(std::max(now, when));
    using namespace std::chrono;
    auto delay = duration_cast<microseconds>(when - now);
    setCancellable(scheduler.schedule(action, delay));
}

void PlaylistUpdater::setScheduled(TimePoint timePoint)
{
    MediaRequest::cancel();
    m_scheduled = timePoint;
}
}
}
