#pragma once

#include "playercore/MediaTime.hpp"
#include <chrono>
#include <limits>
#include <map>
#include <memory>
#include <string>
#include <vector>

namespace twitch {
namespace hls {
class Segment {
public:
    /** Represents attributes from a EXT-X-KEY tag  */
    struct Encryption {
        enum EncryptionMethod {
            NONE,
            AES_128,
            SAMPLE_AES,
            COMMON_CENC,
        };

        EncryptionMethod method = NONE;
        std::string uri;
        std::string keyformat;
        std::vector<uint8_t> iv;
        std::vector<uint8_t> key;
        // KEYFORMATVERSIONS unsupported
    };

    static constexpr int InvalidSequenceNumber = -1;
    using ProgramTime = std::chrono::system_clock::time_point;
    static ProgramTime ProgramTimeNone;
    static constexpr double DateRangeInfinite = std::numeric_limits<double>::infinity();

    /** Represents attributes from a EXT-X-DATERANGE tag  */
    struct DateRange {
        ProgramTime start = ProgramTimeNone;
        ProgramTime end = ProgramTimeNone;
        std::string id;
        double duration = 0.0;
        std::map<std::string, std::string> attributes;
        bool endOnNext;
    };

    std::string url;
    int sequenceNumber = InvalidSequenceNumber;
    bool isInitialization = false;
    std::shared_ptr<Segment> initializationSegment;
    bool discontinuity = false;
    bool start = false;
    MediaTime startOffset;
    MediaTime duration;
    MediaTime cumulativeDuration;
    bool prefetch = false;
    int rangeOffset = -1;
    int rangeLength = -1;
    ProgramTime programDateTime = ProgramTimeNone;
    std::vector<std::shared_ptr<DateRange>> dateRanges;

    Segment() = default;
    // Does this segment immediately follow 'prev'?
    bool isNextAfter(const Segment& prev) const;
    std::string getAbsoluteUrl(const std::string& baseUrl) const;
    std::string getInitializationSegmentUrl(const std::string& baseUrl) const;
};
}
}
