#include "SegmentRequest.hpp"

namespace twitch {
namespace hls {
SegmentRequest::SegmentRequest()
    : MediaRequest(MediaRequest::Type::Segment)
    , m_sequenceNumber(Segment::InvalidSequenceNumber)
    , m_bitrate(0)
    , m_discontinuityAdaptive(true)
    , m_discontinuitySeek(false)
    , m_probe(false)
    , m_rendition(Rendition::Video)
{
}

bool SegmentRequest::isDiscontinuitySeek() const
{
    return m_discontinuitySeek;
}

bool SegmentRequest::isDiscontinuitySequence() const
{
    if (m_lastSegment.isInitialization || m_segment.isInitialization) {
        return false;
    }

    bool hadSegment = m_lastSegment.sequenceNumber >= 0;
    return (hadSegment && !m_segment.isNextAfter(m_lastSegment));
}

bool SegmentRequest::isDiscontinuityInitialization(const std::string& baseUrl) const
{
    if (m_lastSegment.isInitialization) {
        return false;
    }

    // We should check the byte-range as well
    auto newInit = m_segment.getInitializationSegmentUrl(baseUrl);
    auto oldInit = m_lastSegment.getInitializationSegmentUrl(baseUrl);
    return isDiscontinuityAdaptive() || (newInit != oldInit);
}

bool SegmentRequest::isDiscontinuityAdaptive() const
{
    return m_discontinuityAdaptive;
}

bool SegmentRequest::isMediaPrefetch() const
{
    // probe segments are completed
    return m_segment.prefetch && !m_probe;
}

void SegmentRequest::onDiscontinuity()
{
    if (m_reader) {
        using namespace twitch::media;
        uint32_t flags = MediaReader::DiscontinuityNone;
        if (m_segment.discontinuity) {
            flags |= MediaReader::DiscontinuityFormat;
        }
        if (isDiscontinuitySeek()) {
            flags |= MediaReader::DiscontinuitySeek;
        }
        if (isDiscontinuitySequence()) {
            flags |= MediaReader::DiscontinuitySequence;
        }
        if (isDiscontinuityAdaptive()) {
            flags |= MediaReader::DiscontinuityAdaptive;
        }
        // don't apply empty flags
        if (flags != MediaReader::DiscontinuityNone) {
            m_reader->onDiscontinuity(flags);
        }
    }
    m_discontinuitySeek = false;
    m_discontinuityAdaptive = false;
}

void SegmentRequest::setSegment(const Segment& segmentInfo)
{
    if (isPending()) {
        cancel();
    }
    m_id = MediaRequest::generateRequestId();
    m_segment = segmentInfo;
}

void SegmentRequest::addData(const uint8_t* data, size_t size, bool endOfStream)
{
    if (m_reader && getAppendedBytes() == 0) {
        m_segmentStart = m_reader->getDuration();
        m_reader->setDuration(m_segment.duration);
    }

    if (m_reader) {
        // parse data
        m_reader->addData(data, size, endOfStream);
        // update media duration
        setMediaDuration(m_reader->getDuration() - m_segmentStart);
    }

    MediaRequest::appendedBytes(size, endOfStream);
}

void SegmentRequest::onCompleted()
{
    MediaRequest::onCompleted();
    m_lastSegment = m_segment;
    if (!m_segment.isInitialization) {
        m_sequenceNumber++;
    }
}

void SegmentRequest::onVariantChange(bool adaptive)
{
    m_discontinuityAdaptive = true;

    if (!adaptive) {
        cancel();
        if (m_reader) {
            m_reader->reset();
        }
    }
}

void SegmentRequest::onSeek()
{
    cancel();
    m_discontinuitySeek = true;
    m_discontinuityAdaptive = true;
}

void SegmentRequest::onRequest(std::shared_ptr<HttpRequest> request)
{
    MediaRequest::onRequest(request);

    // apply segment range info
    if (m_segment.rangeOffset >= 0) {
        auto start = m_segment.rangeOffset + m_appendedBytes;
        std::string range("bytes=" + std::to_string(start) + "-");
        if (m_segment.rangeLength >= 0) {
            range += std::to_string(start + m_segment.rangeLength - 1);
        }
        request->setHeader("Range", range);
    }
}

void SegmentRequest::onResponse(const HttpResponse& response)
{
    MediaRequest::onResponse(response);
}
}
}
