#include "CEACaptions.hpp"
#include "avc/avcutil.hpp"
#include "debug/trace.hpp"
#include "fourcc.hpp"
#include <string>

namespace twitch {
namespace media {
CEACaptions::CEACaptions(std::function<void(std::shared_ptr<MediaSampleBuffer>)> onFrame)
    : m_onFrame(std::move(onFrame))
{
    reset();
}

CEACaptions::~CEACaptions()
{
    reset();
}

void CEACaptions::reset()
{
    caption_frame_init(&m_captionFrame);
}

void CEACaptions::fromMediaSampleBuffer(const std::shared_ptr<MediaSampleBuffer>& avc)
{
    for (auto nalu : NalIterator(avc->buffer)) {
        if (AVCParser::NalTypeSEI == nalu.type) {
            double dts_seconds = avc->decodeTime.seconds();
            double pts_seconds = avc->presentationTime.seconds();

            if (LIBCAPTION_READY == avcnalu_to_caption_frame(&m_captionFrame, nalu.data, nalu.size, dts_seconds, pts_seconds - dts_seconds)) {
                int lines = static_cast<int>(m_captionFrame.state.rup ? 1 + m_captionFrame.state.rup : 0);
                std::string captions(CAPTION_FRAME_TEXT_BYTES, '\0');
                captions.resize(caption_frame_to_text(&m_captionFrame, (utf8_char_t*)captions.data()));

                auto obj = json11::Json::object { { "caption",
                    json11::Json::object {
                        { "format", "eia608" },
                        { "rollUp", lines },
                        { "text", captions },
                        { "timestamp", m_captionFrame.timestamp },
                    } } };

                auto json_str = json11::Json(obj).dump();
                auto frame = std::make_shared<MediaSampleBuffer>();
                frame->decodeTime = MediaTime(static_cast<int64_t>(m_captionFrame.timestamp * std::milli::den), std::milli::den);
                frame->presentationTime = frame->decodeTime;
                frame->duration = MediaTime(0, std::milli::den);
                frame->isSyncSample = true;
                frame->buffer.assign(json_str.begin(), json_str.end());
                frame->type = fourcc("json");
                m_onFrame(frame);
            }
        }
    }
}
}
}
