#include "CodecString.hpp"
#include <algorithm>
#include <cstdlib>

namespace twitch {
namespace media {
const std::string CodecString::avc1 = "avc1";
const std::string CodecString::mp4a = "mp4a";
const std::string CodecString::vp09 = "vp09";

uint8_t parseInt(const std::string& input, int base = 10)
{
    return static_cast<uint8_t>(strtol(input.c_str(), nullptr, base));
}

CodecString CodecString::parse(const std::string& codecs)
{
    std::vector<std::string> values;
    split(codecs, values, ',');

    CodecString codecString;

    for (const auto& codec : values) {
        auto position = codec.find_first_of('.');
        if (position == std::string::npos) {
            continue;
        }
        std::string type = codec.substr(0, position);
        codecString.values[type] = codec.substr(position + 1, codec.size());

        if (type == avc1) {
            // hack to work around broken twitch vods with incorrect codec data
            // https://jira.twitch.com/browse/VI-2984
            if (codecString.values[avc1] == "000000") {
                codecString.values[avc1] = "42001e";
            }
            // avc parameters need to be converted for the web, so parse in advance
            AVCParameters parameters {};
            codecString.parseAVCParameters(parameters);
        }
    }

    return codecString;
}

MediaType CodecString::getMediaType(const std::string& type)
{
    MediaType mediaType;
    if (type == CodecString::avc1) {
        mediaType = MediaType::Video_AVC;
    } else if (type == CodecString::vp09) {
        mediaType = MediaType::Video_VP9;
    } else if (type == CodecString::mp4a) {
        mediaType = MediaType::Audio_AAC;
    }
    return mediaType;
}

void CodecString::parseAVCParameters(AVCParameters& parameters)
{
    const auto& codec = values[avc1];

    if (codec.empty()) {
        return;
    }
    // profile is the initial byte e.g. avc1.4D402A -> 77 or also could be avc1.77.31 -> 77
    auto nextDot = codec.find_first_of('.', 0);
    int base = nextDot == std::string::npos ? 16 : 10;

    size_t profileLength = 2;
    if (base == 10) {
        profileLength = nextDot;
    }
    std::string profileIdc = codec.substr(0, profileLength);
    parameters.profile = parseInt(profileIdc, base);
    // constraint flags
    if (base == 16) {
        std::string constraints = codec.substr(2, 2);
        parameters.constraintFlags = parseInt(constraints, base);
    } else {
        parameters.constraintFlags = 0;
    }
    // level
    size_t levelStart = 4;
    if (base == 10) {
        levelStart = codec.find_first_of('.', nextDot) + 1;
    }
    std::string levelIdc = codec.substr(levelStart, 2);
    parameters.level = parseInt(levelIdc, base);

    // this checks the reserved bits of the constraint flags are valid (https://tools.ietf.org/html/rfc6381#section-3.3)
    // some twitch vods have invalid flags that have to be corrected e.g. https://www.twitch.tv/videos/173765787
    bool invalidConstraintFlags = (parameters.constraintFlags & 0x00000003) != 0;
    if (invalidConstraintFlags) {
        parameters.constraintFlags = 0;
    }

    if (base == 10 || invalidConstraintFlags) {
        // convert to hex this is needed for the mime type check on the web
        char buffer[7] = { 0 };
        std::snprintf(buffer, 7, "%02X%02X%02X", parameters.profile, parameters.constraintFlags, parameters.level);
        values[avc1] = buffer;
    }
}

void CodecString::parseVP9Parameters(VP9Parameters& parameters)
{
    const auto& codec = values[vp09];
    std::vector<std::string> parts;
    split(codec, parts, '.');

    //https://www.webmproject.org/vp9/mp4/#codecs-parameter-string
    if (parts.size() >= 3) {
        size_t index = 0;
        parameters.profile = parseInt(parts[index++]);
        parameters.level = parseInt(parts[index++]);
        parameters.bitDepth = parseInt(parts[index++]);

        if (index < parts.size()) {
            parameters.chromaSubsampling = parseInt(parts[index++]);
        }
        if (index < parts.size()) {
            parameters.colourPrimaries = parseInt(parts[index++]);
        }
        if (index < parts.size()) {
            parameters.transferCharacteristics = parseInt(parts[index++]);
        }
        if (index < parts.size()) {
            parameters.matrixCoefficients = parseInt(parts[index++]);
        }
        if (index < parts.size()) {
            parameters.videoFullRangeFlag = parseInt(parts[index++]);
        }
    }
}

void CodecString::split(const std::string& input, std::vector<std::string>& values, char delimiter)
{
    if (!input.empty()) {
        std::string::size_type start = 0;
        std::string::size_type end = 0;

        while ((end = input.find(delimiter, start)) != std::string::npos) {
            values.push_back(input.substr(start, end - start));
            start = end + 1;
        }

        values.push_back(input.substr(start, end - start));
    }
}

std::string CodecString::format() const
{
    std::string codecs;
    for (const auto& entry : values) {
        if (!codecs.empty()) {
            codecs += ",";
        }
        codecs += entry.first + "." + entry.second;
    }
    return codecs;
}
}
}
