#include "FragmentReader.hpp"
#include "FragmentSample.hpp"
#include "MemoryStream.hpp"
#include "SourceFormat.hpp"
#include "id3/id3.hpp"

namespace twitch {
namespace media {
FragmentReader::FragmentReader(const Platform& platform, MediaReader::Listener& listener,
    const MediaType& mediaType, TrackId trackId, const std::string& path, bool outputPartialTS)
    : m_mediaType(mediaType)
    , m_isFragmentedMP4(mediaType.matches(MediaType::Video_MP4))
    , m_outputPartial(outputPartialTS)
    , m_trackId(trackId)
    , m_path(path)
    , m_listener(listener)
    , m_log(platform.getLog())
    , m_discontinuity(true)
    , m_syncSample(true)
{
    m_ceaCaptions.reset(new CEACaptions([this](const std::shared_ptr<MediaSampleBuffer>& sample) {
        createMetadataTrack();
        // compensate for timestamp offset TODO better timestamp handling
        MediaTime baseDecodeTime = getFragmentDecodeTime();
        sample->decodeTime -= baseDecodeTime;
        sample->presentationTime -= baseDecodeTime;
        sample->decodeTime += m_cumulativeDuration;
        sample->presentationTime += m_cumulativeDuration;
        m_listener.onMediaSample(MediaReader::MetaTrackId, sample);
    }));

    reset();
}

void FragmentReader::seekTo(MediaTime time)
{
    m_cumulativeDuration = time;
    m_discontinuity = true;

    if (m_tsparser) {
        m_tsparser->seek(time.scaleTo(TSTimeScale).count());
    }
    m_lastOutputDuration = time;

    resetParserStream();
}

void FragmentReader::addData(const uint8_t* data, size_t size, bool endOfStream)
{
    if (size) {
        m_stream->seek(static_cast<size_t>(m_stream->length()));
        m_stream->write(data, size);
    }

    // fragmented MP4 incremental parsing mode
    if (m_isFragmentedMP4) {
        while (m_mp4parser->canReadTracks()) {
            m_mp4parser->readTracks();

            if (m_mp4parser->isInitializationFragment()) {
                // don't need to incrementally parse init fragment
                break;
            }

            auto nextFragmentOffset = static_cast<int64_t>(m_mp4parser->getNextFragmentOffset());
            bool hasNextFragment = nextFragmentOffset > 0 && nextFragmentOffset < m_stream->length();

            if (hasNextFragment) {
                outputFragment(nextFragmentOffset, endOfStream);
            } else {
                // don't have enough data
                break;
            }
        }
    } else if (m_outputPartial) {
        // ts output partial segments mode
        if (m_tsparser) {
            m_tsparser->addData(data, size);
            outputFragment(m_stream->length(), endOfStream);
        }
    }

    if (endOfStream) {
        if (m_stream->length()) {
            outputFragment(m_stream->length(), true);
        }
        resetParserStream();
    }
}

void FragmentReader::onDiscontinuity(uint32_t flags)
{
    // if just the DiscontinuityVariant flag is set don't consider it a discontinuity
    if (flags != MediaReader::DiscontinuityAdaptive && flags != MediaReader::DiscontinuityNone) {
        m_discontinuity = true;
    }
}

void FragmentReader::reset()
{
    if (m_isFragmentedMP4) {
        m_mp4parser.reset(new Mp4Parser(m_log));
    } else if (m_outputPartial) {
        m_tsparser.reset(new TransportStream(*this));
    }
    m_discontinuity = true;
    m_lastOutputDuration = MediaTime::zero();
    resetParserStream();
}

void FragmentReader::setDuration(MediaTime duration)
{
    m_mediaDuration = duration;
}

void FragmentReader::setStream(std::unique_ptr<Stream> stream)
{
    m_stream = std::move(stream);
}

void FragmentReader::readSamples(MediaTime duration)
{
    // not supported
    (void)duration;
}

void FragmentReader::outputFragment(int64_t endPosition, bool endOfStream)
{
    MediaTime duration = getOutputDuration();
    bool isInitialization = m_mp4parser && m_mp4parser->isInitializationFragment();

    if (m_discontinuity || isInitialization) {
        std::shared_ptr<SourceFormat> format = std::make_shared<SourceFormat>(m_mediaType);
        format->setName(m_quality);
        format->setPath(m_path);
        if (m_isFragmentedMP4 && !m_mp4parser->getPsshBytes().empty()) {
            format->setProtectionData(m_mp4parser->getPsshBytes());
        }
        m_formats[m_trackId] = format;
        m_listener.onMediaTrack(m_trackId, format);
    }

    // twitch specific metadata
    if (m_mp4parser && !m_mp4parser->getEmsgs().empty()) {
        for (const auto& emsg : m_mp4parser->getEmsgs()) {
            if (emsg.scheme_id_uri == "urn:twitch:id3") {
                // content is ID3 binary like TS
                MediaTime offset(emsg.presentation_time_delta, emsg.timescale);

                auto frames = Id3::parseFrames(emsg.data, m_cumulativeDuration + offset);
                createMetadataTrack();
                for (const auto& frame : frames) {
                    m_listener.onMediaSample(MediaReader::MetaTrackId, frame);
                }
            }
        }
    }
    if (m_mp4parser && !isInitialization) {
        if (m_metaTracks.empty()) {
            for (const auto& track : m_mp4parser->getTracks()) {
                int codec = track->getCodecBoxType();
                if (codec == MP4_avc1 || codec == MP4_encv || codec == MP4_wvtt) {
                    m_metaTracks.push_back(track);
                    break;
                }
            }
        }

        if (!m_metaTracks.empty()) {
            m_mp4parser->readSamples(
                m_mp4parser->getTracks(),
                [this](const Mp4Track& track, const std::shared_ptr<MediaSampleBuffer>& sample) {
                    // read CEA 608/708 captions from the avc track if present
                    int codec = track.getCodecBoxType();
                    if (codec == MP4_avc1 || codec == MP4_encv) {
                        m_ceaCaptions->fromMediaSampleBuffer(sample);
                    } else if (codec == MP4_wvtt) {
                        createWebVTTTrack();
                        sample->type = MP4_wvtt;
                        m_listener.onMediaSample(MediaReader::TextTrackId, sample);
                    }
                },
                MediaTime::max());
        }
    }

    MediaTime decodeTime = getFragmentDecodeTime();
    auto sample = std::make_shared<FragmentSample>();
    auto memoryStream = static_cast<MemoryStream*>(m_stream.get());
    sample->buffer = memoryStream->take(endPosition);
    sample->decodeTime = decodeTime;
    sample->presentationTime = m_cumulativeDuration;
    sample->duration = duration;
    sample->isSyncSample = m_syncSample;
    sample->isDiscontinuity = m_discontinuity;
    sample->isInitialization = isInitialization;
    sample->isEndOfStream = endOfStream;
    sample->mediaDuration = m_mediaDuration; // duration of whole segment

    m_cumulativeDuration += duration;

    // reset discontinuity flag
    if (m_discontinuity && !isInitialization) {
        m_discontinuity = false;
    }

    // next sync sample will be marked from the addData callback
    m_syncSample = false;

    m_listener.onMediaSample(m_trackId, sample);
    m_listener.onMediaFlush();
}

void FragmentReader::resetParserStream()
{
    // arbitrary limit just to prevent unlimited data growth
    m_stream.reset(new MemoryStream());
    m_syncSample = true;

    if (m_mp4parser) {
        m_mp4parser->setStream(m_stream.get());
    }
}

MediaTime FragmentReader::getFragmentDecodeTime()
{
    MediaTime decodeTime = m_cumulativeDuration;
    if (m_isFragmentedMP4 && !m_mp4parser->isInitializationFragment()) {
        MediaTime fragmentDecodeTime;
        // count the tracks with media data
        int mediaTrackCount = 0;
        for (const auto& track : m_mp4parser->getTracks()) {
            if (track->getHandlerType() == MP4_vide || track->getHandlerType() == MP4_soun) {
                mediaTrackCount++;
            }
        }

        for (const auto& track : m_mp4parser->getTracks()) {
            // if muxed MP4 video fragment used only the video track
            if (mediaTrackCount == 1 || track->getHandlerType() == MP4_vide) {
                MediaTime baseMediaDecodeTime(track->getBaseMediaDecodeTime(), track->getTimescale());
                fragmentDecodeTime = std::max(fragmentDecodeTime, baseMediaDecodeTime);
            }
        }
        decodeTime = fragmentDecodeTime;
    }
    return decodeTime;
}

MediaTime FragmentReader::getOutputDuration()
{
    MediaTime duration;
    if (m_tsparser) {
        MediaTime tsDuration = MediaTime(m_tsparser->duration(), TSTimeScale);
        MediaTime buffered = tsDuration - m_lastOutputDuration;
        duration = buffered;
        m_lastOutputDuration = tsDuration;
    } else if (m_mp4parser) {
        duration = m_mp4parser->getDuration();
    } else {
        duration = m_mediaDuration;
    }
    return duration;
}

void FragmentReader::createMetadataTrack()
{
    if (!m_formats.count(MediaReader::MetaTrackId)) {
        auto format = std::make_shared<SourceFormat>(MediaType::Text_Json);
        m_formats[MediaReader::MetaTrackId] = format;
        m_listener.onMediaTrack(MediaReader::MetaTrackId, format);
    }
}

void FragmentReader::createWebVTTTrack()
{
    if (!m_formats.count(MediaReader::TextTrackId)) {
        auto format = std::make_shared<SourceFormat>(MediaType::Text_VTT);
        m_formats[MediaReader::TextTrackId] = format;
        m_listener.onMediaTrack(MediaReader::TextTrackId, format);
    }
}

std::shared_ptr<const MediaFormat> FragmentReader::getTrackFormat(TrackId id)
{
    return m_formats[id];
}

void FragmentReader::onElementaryDiscontinuity(uint8_t type)
{
    (void)type;
}

void FragmentReader::onElementarySample(uint8_t type, const std::shared_ptr<MediaSampleBuffer>& sample)
{
    (void)type;
    (void)sample;
}
}
}
