#include "playercore/MediaTime.hpp"
#include <cassert>
#include <cstdlib>
#include <limits>

namespace twitch {
using std::chrono::duration;
using std::chrono::duration_cast;
const uint32_t DefaultScale = std::micro::den;

MediaTime::MediaTime() noexcept
    : MediaTime(0, DefaultScale)
{
}

MediaTime::MediaTime(int64_t value, uint32_t scale) noexcept
    : m_value(value)
    , m_scale(scale)
{
}

MediaTime::MediaTime(double seconds) noexcept
    : MediaTime(static_cast<int64_t>(seconds * DefaultScale), DefaultScale)
{
}

MediaTime MediaTime::zero()
{
    return MediaTime(0, DefaultScale);
}

MediaTime MediaTime::max()
{
    return MediaTime(std::numeric_limits<int64_t>::max(), DefaultScale);
}

MediaTime MediaTime::invalid()
{
    return MediaTime(-1, 0);
}

int MediaTime::compare(const MediaTime& rhs) const
{
    if (m_scale == rhs.m_scale) {
        if (m_scale == 0 && rhs.m_scale == 0) {
            return 0;
        }
        if (m_value > rhs.m_value) {
            return 1;
        } else if (m_value < rhs.m_value) {
            return -1;
        }
        return 0;
    } else {
        return compare(rhs.scaleTo(m_scale));
    }
}

MediaTime MediaTime::scaleTo(uint32_t scale) const
{
    if (scale == m_scale) {
        return MediaTime(m_value, m_scale);
    } else if (valid() && (scale % m_scale == 0)) {
        return MediaTime(m_value * (scale / m_scale), scale);
    }
    return MediaTime(static_cast<int64_t>(seconds() * scale), scale);
}

MediaTime MediaTime::absolute() const
{
    return MediaTime(std::abs(m_value), m_scale);
}

bool MediaTime::valid() const
{
    return m_scale > 0;
}

double MediaTime::seconds() const
{
    if (valid()) {
        return m_value / static_cast<double>(timebase());
    } else {
        assert(false);
        return std::numeric_limits<double>::max();
    }
}

MediaTime::Milliseconds MediaTime::milliseconds() const
{
    if (m_scale == std::milli::den) {
        return Milliseconds(m_value);
    }
    return duration_cast<Milliseconds>(duration<double>(seconds()));
}

MediaTime::Microseconds MediaTime::microseconds() const
{
    if (m_scale == std::micro::den) {
        return Microseconds(m_value);
    }
    return duration_cast<Microseconds>(duration<double>(seconds()));
}

MediaTime::Nanoseconds MediaTime::nanoseconds() const
{
    if (m_scale == std::nano::den) {
        return Nanoseconds(m_value);
    }
    return duration_cast<Nanoseconds>(duration<double>(seconds()));
}

MediaTime& MediaTime::operator*=(double rhs)
{
    m_value = static_cast<int64_t>(m_value * rhs);
    return *this;
}

MediaTime& MediaTime::operator/=(double rhs)
{
    m_value = static_cast<int64_t>(m_value / rhs);
    return *this;
}

MediaTime& MediaTime::operator+=(const MediaTime& rhs)
{
    if (rhs.m_scale == m_scale) {
        m_value += rhs.m_value;
    } else {
        m_value += rhs.scaleTo(m_scale).m_value;
    }
    return *this;
}

MediaTime& MediaTime::operator-=(const MediaTime& rhs)
{
    if (rhs.m_scale == m_scale) {
        m_value -= rhs.m_value;
    } else {
        m_value -= rhs.scaleTo(m_scale).m_value;
    }
    return *this;
}
}
