#include "MediaType.hpp"
#include <algorithm>
#include <locale>
#include <vector>

namespace twitch {
const std::string MediaType::Type_Applicaton("application");
const std::string MediaType::Type_Audio("audio");
const std::string MediaType::Type_Video("video");
const std::string MediaType::Type_Text("text");

const MediaType MediaType::Application_OctetStream("application", "octet-stream");
const MediaType MediaType::Application_Apple_MPEG_URL("application", "vnd.apple.mpegurl");
const MediaType MediaType::Application_MPEG_URL("application", "x-mpegURL");
const MediaType MediaType::Application_Json("application", "json");
const MediaType MediaType::Binary_OctetStream("binary", "octet-stream");
const MediaType MediaType::Video_AVC("video", "avc");
const MediaType MediaType::Audio_AAC("audio", "aac");
const MediaType MediaType::Audio_MP4("audio", "mp4");
const MediaType MediaType::Audio_PCM("audio", "pcm");
const MediaType MediaType::Video_MP4("video", "mp4");
const MediaType MediaType::Video_MP2T("video", "MP2T");
const MediaType MediaType::Video_NV12("video", "nv12");
const MediaType MediaType::Video_NV21("video", "nv21");
const MediaType MediaType::Video_Quicktime("video", "quicktime");
const MediaType MediaType::Video_VP9("video", "x-vnd.on2.vp9");
const MediaType MediaType::Text_VTT("text", "vtt");
const MediaType MediaType::Text_ID3("text", "id3");
const MediaType MediaType::Text_Json("text", "json");
const MediaType MediaType::Text_Plain("text", "plain");

MediaType::MediaType(const std::string& mediaType)
    : name(mediaType)
{
    auto position = mediaType.find_first_of('/');
    if (position != std::string::npos) {
        type = mediaType.substr(0, position);
        auto lower = [](char c) { return std::tolower(c, std::locale()); };
        std::transform(type.begin(), type.end(), type.begin(), lower);
        auto remaining = mediaType.substr(position + 1);

        position = remaining.find_first_of(';');
        if (position == std::string::npos) {
            subType = remaining;
        } else {
            subType = remaining.substr(0, position);
            parameters = remaining.substr(position + 1, remaining.size());
        }
        std::transform(subType.begin(), subType.end(), subType.begin(), lower);
    }
}

MediaType::MediaType(const std::string& type, const std::string& subType, const std::string& parameters)
    : name(type + "/" + subType + (parameters.empty() ? "" : ";" + parameters))
    , type(type)
    , subType(subType)
    , parameters(parameters)
{
}

bool MediaType::matches(const MediaType& other) const
{
    return (type == other.type || type == "*") && (subType == other.subType || subType == "*");
}

MediaType MediaType::withoutParameters() const
{
    return MediaType(type, subType);
}

MediaType MediaType::matchFromPath(const std::string& path)
{
    // strip the path
    std::string stripped = path;
    auto position = path.find_last_of('/');

    if (position != std::string::npos) {
        stripped = path.substr(position + 1);
        for (char trim : { '?', '#' }) {
            position = stripped.find(trim);

            if (position != std::string::npos) {
                stripped = stripped.substr(0, position);
            }
        }
    }

    auto endsWith = [](const std::string& input, const std::string& extension) {
        return input.length() > extension.length()
            && input.compare(input.length() - extension.length(), extension.length(), extension) == 0;
    };

    // check if this url or path is probably an mp4 file
    if (endsWith(stripped, ".mp4") || endsWith(stripped, ".3gp")) {
        return MediaType::Video_MP4;
    } else if (endsWith(stripped, ".ts")) {
        return MediaType::Video_MP2T;
    } else if (endsWith(stripped, ".m3u8")) {
        return MediaType::Application_MPEG_URL;
    } else if (endsWith(stripped, ".mov")) {
        return MediaType::Video_Quicktime;
    }

    return MediaType();
}
}
