#pragma once

#include <cstdint>
#include <cstring>
#include <vector>

namespace twitch {
/** Represents a stream of bytes */
class Stream {
public:
    virtual ~Stream() = default;

    /**
     * Seek in the stream
     * @param position to seek to, between 0 and the length() of the stream
     * @return true if the seek was successful, otherwise false
     */
    virtual bool seek(size_t position) = 0;

    /**
     * Read bytes from the stream.
     * @param buffer destination buffer
     * @param size maximum bytes to copy
     * @return the number of bytes copied successfully into buffer, or -1 if stream is in an error state
     */
    virtual int64_t read(uint8_t* buffer, size_t size) = 0;

    /**
     * Write bytes to the stream.
     * @param buffer source buffer
     * @param size of source buffer
     * @return the number of bytes written into the stream if successful, or -1 if stream is in an error state
     */
    virtual int64_t write(const uint8_t* buffer, size_t size) = 0;

    /** @return the total length of the stream in bytes, or -1 if stream is in an error state */
    virtual int64_t length() const = 0;

    /** @return current position in the stream, or -1 if stream is in an error state */
    virtual int64_t position() const = 0;

    /**
     * Flushes the stream
     * @return true if the stream flushed successfully, otherwise false
     */
    virtual bool flush() = 0;

    /**
     * Close the stream. This is typically done automatically by the destructor.
     */
    virtual void close() = 0;

    /** @return 0 if no error exists, otherwise a non-zero error code */
    virtual int error() const = 0;
};
}
