#include "aacutil.hpp"
#include <assert.h>
#include <stdio.h>
#include <string.h>

const unsigned aac_frequency_table[16] = { 96000, 88200, 64000, 48000, 44100, 32000, 24000, 22050, 16000, 12000, 11025, 8000, 7350, 0, 0, 0 };
unsigned aac_frequency_from_index(unsigned index) { return 15 < index ? 0 : aac_frequency_table[index]; }

void aac_init_adts(adts_t* adts)
{
    memset(adts, 0, sizeof(adts_t));
}

int aac_parse_adts(adts_t* adts, const uint8_t* data, size_t size)
{
    if (AAC_DEFAULT_HEADER_SIZE > size) {
        aac_init_adts(adts);
        return 0;
    }

    adts->syncWord = (data[0] << 4) | (data[1] & 0xF0) >> 4;
    adts->protectionAbsent = (data[1] & 0x01);
    adts->objectType = ((data[2] & 0xC0) >> 6) + 1;
    adts->frequencyIndex = ((data[2] & 0x3C) >> 2);
    adts->channelConfig = ((data[2] & 0x01) << 2) | ((data[3] & 0xC0) >> 6);
    adts->frameSize = ((data[3] & 0x03) << 11) | ((data[4] & 0xFF) << 3) | ((data[5] & 0xE0) >> 5);
    adts->bufferSize = ((data[5] & 0x1F) << 6) | ((data[6] & 0xFC) >> 2);
    adts->frameCount = ((data[6] & 0x03) + 1);
    adts->sampleCount = adts->frameCount * AAC_SAMPLES_PER_FRAME; // TODO mp3 is different!

    if (!adts->protectionAbsent && 9 <= size) {
        adts->crc = (data[7] << 8) | data[8];
    } else {
        adts->crc = 0;
    }

    return 1;
}

int aac_parse_extradata(adts_t* adts, const uint8_t* data, size_t size)
{
    if (2 > size) {
        return 0;
    }

    adts->syncWord = AAC_SYNC_WORD;
    adts->protectionAbsent = 1;
    adts->frameSize = adts->protectionAbsent ? AAC_DEFAULT_HEADER_SIZE : AAC_EXTENDED_HEADER_SIZE;
    adts->frameCount = 0;
    adts->sampleCount = 0;
    adts->crc = 0;
    // AAAA ABBB BCCC CXXX
    adts->objectType = (data[0] & 0xF8) >> 3;
    adts->frequencyIndex = ((data[0] & 0x07) << 1) | ((data[1] & 0x80) >> 7);
    adts->channelConfig = (data[1] & 0x78) >> 3;

    return 1;
}

// https://wiki.multimedia.cx/index.php/ADTS
int aac_render_adts(const adts_t* adts, uint8_t* data)
{
    data[0] = 0xFF;
    data[1] = 0xF1;
    data[2] = ((adts->objectType - 1) & 0x03) << 6
        | (adts->frequencyIndex & 0x0F) << 2
        | (adts->channelConfig & 0x40) >> 2;
    data[3] = (adts->channelConfig & 0x02) << 6
        | (uint8_t)((adts->frameSize & 0x1800) >> 11);
    data[4] = (uint8_t)(adts->frameSize >> 3);
    data[5] = (uint8_t)((adts->frameSize & 0x07) << 5) | 0x1F;
    data[6] = 0xFC | ((adts->frameCount - 1) & 0x03);
    return 1;
}

int aac_render_extradata(const adts_t* adts, uint8_t* data, size_t size)
{
    if (2 > size) {
        return 0;
    }

    data[0] = ((adts->objectType & 0x1F) << 3) | ((adts->frequencyIndex & 0x0F) >> 1);
    data[1] = ((adts->frequencyIndex & 0x01) << 7) | ((adts->channelConfig & 0x0F) << 3);
    return 1;
}

int aac_channels(const adts_t* adts)
{
    switch (adts->channelConfig) {
    case 1:
    case 2:
    case 3:
    case 4:
    case 5:
    case 6:
        return adts->channelConfig;

    case 7:
        return 1 + adts->channelConfig;

    default:
        return 0;
    }
}

int aac_element_instance_tag(const uint8_t* data)
{
    switch (0xE0 & (*data)) {
    case 0x00: // single_channel_element()
    case 0x20: // channel_pair_element()
        return ((*data) >> 1) & 0x0F;

    default:
        return -1;
    }
}

enum aacSanity aac_sanity(const adts_t* adts)
{
    if (AAC_SYNC_WORD != adts->syncWord) {
        return kAacInvalidSyncWord;
    }

    // framesize includes the header, so it can not be smaller
    if (adts->frameSize < aac_header_size(adts)) {
        return kAacFrameSmallerThanHeader;
    }

    if (!aac_frequency(adts)) {
        return kAacInvalidFrequency;
    }

    if (0 >= aac_channels(adts)) {
        return kAacInvalidChannelValue;
    }

    if (0 >= adts->frameCount || 4 < adts->frameCount) {
        return kAacInvalidFrameCount;
    }

    // TODO: causing problems on 1.1, fix later
    // unsigned min_size = ((aac_channels(adts) * AAC_MIN_BITRATE * AAC_SAMPLES_PER_FRAME) / (aac_frequency(adts) * 8) / adts->frameCount);

    // if ((adts->frameSize - aac_header_size(adts)) < min_size) {
    //     return kAacInvalidFrameSize_TooSmall;
    // }

    // unsigned max_size = ((aac_channels(adts) * AAC_MAX_BITRATE * AAC_SAMPLES_PER_FRAME) / (aac_frequency(adts) * 8) / adts->frameCount);
    // if ((adts->frameSize - aac_header_size(adts)) > max_size) {
    //     return kAacInvalidFrameSize_TooLarge;
    // }

    return kAacOk;
}

// This is a simple rle encoding, first byte is the run lenght minus 1, second byte is the repeating charcter;
static const std::vector<uint8_t> aac_silence_frames_rle[2][16] = { {
                                                                        { 0x00, 0x01, 0x00, 0x40, 0x00, 0x22, 0x00, 0x80, 0x00, 0xa3, 0x00, 0x78, 0x00, 0xb0, 0x00, 0x85, 0x21, 0x2d, 0x00, 0x2f },
                                                                        { 0x00, 0x01, 0x00, 0x40, 0x00, 0x22, 0x00, 0x80, 0x00, 0xa3, 0x00, 0x78, 0x00, 0xc8, 0x00, 0x85, 0x24, 0x2d, 0x00, 0x2f },
                                                                        { 0x00, 0x01, 0x00, 0x40, 0x00, 0x22, 0x00, 0x80, 0x00, 0xa3, 0x00, 0x79, 0x00, 0x58, 0x00, 0x85, 0x36, 0x2d, 0x00, 0x2f },
                                                                        { 0x00, 0x01, 0x00, 0x40, 0x00, 0x22, 0x00, 0x80, 0x00, 0xa3, 0x00, 0x7a, 0x00, 0x00, 0x00, 0x85, 0x4b, 0x2d, 0x00, 0x2f },
                                                                        { 0x00, 0x01, 0x00, 0x40, 0x00, 0x22, 0x00, 0x80, 0x00, 0xa3, 0x00, 0x7a, 0x00, 0x40, 0x00, 0x85, 0x53, 0x2d, 0x00, 0x2f },
                                                                        { 0x00, 0x01, 0x00, 0x40, 0x00, 0x22, 0x00, 0x80, 0x00, 0xa3, 0x00, 0x7b, 0x00, 0x58, 0x00, 0x85, 0x76, 0x2d, 0x00, 0x2f },
                                                                        { 0x00, 0x01, 0x00, 0x40, 0x00, 0x22, 0x00, 0x80, 0x00, 0xa3, 0x00, 0x7c, 0x00, 0xb0, 0x00, 0x85, 0xa1, 0x2d, 0x00, 0x2f },
                                                                        { 0x00, 0x01, 0x00, 0x40, 0x00, 0x22, 0x00, 0x80, 0x00, 0xa3, 0x00, 0x7d, 0x00, 0x28, 0x00, 0x85, 0xb0, 0x2d, 0x00, 0x2f },
                                                                        { 0x00, 0x01, 0x00, 0x40, 0x00, 0x22, 0x00, 0x80, 0x00, 0xa3, 0x00, 0x7f, 0x00, 0x58, 0x00, 0x85, 0xf6, 0x2d, 0x00, 0x2f },
                                                                        { 0x00, 0x01, 0x00, 0x40, 0x00, 0x22, 0x00, 0x80, 0x00, 0xa3, 0x00, 0x7f, 0x00, 0xf8, 0x00, 0x85, 0xff, 0x2d, 0x0a, 0x2d, 0x00, 0x2e, 0x00, 0xf3, 0x00, 0x11, 0x00, 0x0a, 0x3c, 0x5a, 0x00, 0x5e },
                                                                        { 0x00, 0x01, 0x00, 0x40, 0x00, 0x22, 0x00, 0x80, 0x00, 0xa3, 0x00, 0x7f, 0x00, 0xf8, 0x00, 0x85, 0xff, 0x2d, 0x0a, 0x2d, 0x00, 0x2e, 0x00, 0xf5, 0x00, 0x01, 0x00, 0x0a, 0x5b, 0x5a, 0x00, 0x5e },
                                                                        { 0x00, 0x01, 0x00, 0x40, 0x00, 0x22, 0x00, 0x80, 0x00, 0xa3, 0x00, 0x7f, 0x00, 0xf8, 0x00, 0x85, 0xff, 0x2d, 0x0a, 0x2d, 0x00, 0x2e, 0x00, 0xfd, 0x00, 0xc1, 0x00, 0x0a, 0xe7, 0x5a, 0x00, 0x5e },
                                                                        { 0x00, 0x01, 0x00, 0x40, 0x00, 0x22, 0x00, 0x80, 0x00, 0xa3, 0x00, 0x7f, 0x00, 0xf8, 0x00, 0x85, 0xff, 0x2d, 0x0a, 0x2d, 0x00, 0x2e, 0x00, 0xfd, 0x00, 0xc1, 0x00, 0x0a, 0xe7, 0x5a, 0x00, 0x5e },
                                                                        { 0x00, 0x01, 0x00, 0x40, 0x00, 0x22, 0x00, 0x80, 0x00, 0xa3, 0x00, 0x7f, 0x00, 0xf8, 0x00, 0x85, 0xff, 0x2d, 0x0a, 0x2d, 0x00, 0x2e, 0x00, 0xfd, 0x00, 0xc1, 0x00, 0x0a, 0xe7, 0x5a, 0x00, 0x5e },
                                                                        { 0x00, 0x01, 0x00, 0x40, 0x00, 0x22, 0x00, 0x80, 0x00, 0xa3, 0x00, 0x7f, 0x00, 0xf8, 0x00, 0x85, 0xff, 0x2d, 0x0a, 0x2d, 0x00, 0x2e, 0x00, 0xfd, 0x00, 0xc1, 0x00, 0x0a, 0xe7, 0x5a, 0x00, 0x5e },
                                                                        { 0x00, 0x01, 0x00, 0x40, 0x00, 0x22, 0x00, 0x80, 0x00, 0xa3, 0x00, 0x7f, 0x00, 0xf8, 0x00, 0x85, 0xff, 0x2d, 0x0a, 0x2d, 0x00, 0x2e, 0x00, 0xfd, 0x00, 0xc1, 0x00, 0x0a, 0xe7, 0x5a, 0x00, 0x5e },
                                                                    },
    {
        { 0x00, 0x21, 0x00, 0x11, 0x00, 0x45, 0x00, 0x00, 0x00, 0x14, 0x00, 0x50, 0x00, 0x01, 0x00, 0x46, 0x00, 0xf3, 0x00, 0xd1, 0x00, 0x0a, 0x48, 0x5a, 0x00, 0x5e },
        { 0x00, 0x21, 0x00, 0x11, 0x00, 0x45, 0x00, 0x00, 0x00, 0x14, 0x00, 0x50, 0x00, 0x01, 0x00, 0x46, 0x00, 0xf4, 0x00, 0x51, 0x00, 0x0a, 0x50, 0x5a, 0x00, 0x5e },
        { 0x00, 0x21, 0x00, 0x11, 0x00, 0x45, 0x00, 0x00, 0x00, 0x14, 0x00, 0x50, 0x00, 0x01, 0x00, 0x46, 0x00, 0xf6, 0x00, 0x81, 0x00, 0x0a, 0x73, 0x5a, 0x00, 0x5e },
        { 0x00, 0x21, 0x00, 0x11, 0x00, 0x45, 0x00, 0x00, 0x00, 0x14, 0x00, 0x50, 0x00, 0x01, 0x00, 0x46, 0x00, 0xf9, 0x00, 0x31, 0x00, 0x0a, 0x9e, 0x5a, 0x00, 0x5e },
        { 0x00, 0x21, 0x00, 0x11, 0x00, 0x45, 0x00, 0x00, 0x00, 0x14, 0x00, 0x50, 0x00, 0x01, 0x00, 0x46, 0x00, 0xfa, 0x00, 0x21, 0x00, 0x0a, 0xad, 0x5a, 0x00, 0x5e },
        { 0x00, 0x21, 0x00, 0x11, 0x00, 0x45, 0x00, 0x00, 0x00, 0x14, 0x00, 0x50, 0x00, 0x01, 0x00, 0x46, 0x00, 0xfe, 0x00, 0x81, 0x00, 0x0a, 0xf3, 0x5a, 0x00, 0x5e },
        { 0x00, 0x21, 0x00, 0x11, 0x00, 0x45, 0x00, 0x00, 0x00, 0x14, 0x00, 0x50, 0x00, 0x01, 0x00, 0x46, 0x00, 0xff, 0x00, 0xf1, 0x00, 0x0a, 0xff, 0x5a, 0x0a, 0x5a, 0x00, 0x5d, 0x00, 0xe5, 0x00, 0xc2, 0x00, 0x14, 0x39, 0xb4, 0x00, 0xbc },
        { 0x00, 0x21, 0x00, 0x11, 0x00, 0x45, 0x00, 0x00, 0x00, 0x14, 0x00, 0x50, 0x00, 0x01, 0x00, 0x46, 0x00, 0xff, 0x00, 0xf1, 0x00, 0x0a, 0xff, 0x5a, 0x0a, 0x5a, 0x00, 0x5d, 0x00, 0xe9, 0x00, 0xa2, 0x00, 0x14, 0x58, 0xb4, 0x00, 0xbc },
        { 0x00, 0x21, 0x00, 0x11, 0x00, 0x45, 0x00, 0x00, 0x00, 0x14, 0x00, 0x50, 0x00, 0x01, 0x00, 0x46, 0x00, 0xff, 0x00, 0xf1, 0x00, 0x0a, 0xff, 0x5a, 0x0a, 0x5a, 0x00, 0x5d, 0x00, 0xfb, 0x00, 0x22, 0x00, 0x14, 0xe4, 0xb4, 0x00, 0xbc },
        { 0x00, 0x21, 0x00, 0x11, 0x00, 0x45, 0x00, 0x00, 0x00, 0x14, 0x00, 0x50, 0x00, 0x01, 0x00, 0x46, 0x00, 0xff, 0x00, 0xf1, 0x00, 0x0a, 0xff, 0x5a, 0x0a, 0x5a, 0x00, 0x5d, 0x00, 0xff, 0x00, 0xe2, 0x00, 0x14, 0xff, 0xb4, 0x0a, 0xb4, 0x00, 0xbb, 0x00, 0xdd, 0x00, 0x44, 0x00, 0x29, 0x80, 0x69, 0x00, 0x78 },
        { 0x00, 0x21, 0x00, 0x11, 0x00, 0x45, 0x00, 0x00, 0x00, 0x14, 0x00, 0x50, 0x00, 0x01, 0x00, 0x46, 0x00, 0xff, 0x00, 0xf1, 0x00, 0x0a, 0xff, 0x5a, 0x0a, 0x5a, 0x00, 0x5d, 0x00, 0xff, 0x00, 0xe2, 0x00, 0x14, 0xff, 0xb4, 0x0a, 0xb4, 0x00, 0xbb, 0x00, 0xec, 0x00, 0x44, 0x00, 0x29, 0xbc, 0x69, 0x00, 0x78 },
        { 0x00, 0x21, 0x00, 0x11, 0x00, 0x45, 0x00, 0x00, 0x00, 0x14, 0x00, 0x50, 0x00, 0x01, 0x00, 0x46, 0x00, 0xff, 0x00, 0xf1, 0x00, 0x0a, 0xff, 0x5a, 0x0a, 0x5a, 0x00, 0x5d, 0x00, 0xff, 0x00, 0xe2, 0x00, 0x14, 0xff, 0xb4, 0x0a, 0xb4, 0x00, 0xbb, 0x00, 0xff, 0x00, 0xc4, 0x00, 0x29, 0xff, 0x69, 0x0a, 0x69, 0x00, 0x77, 0x00, 0xdd, 0x00, 0x88, 0x00, 0x52, 0xc6, 0xd2, 0x00, 0xf0 },
        { 0x00, 0x21, 0x00, 0x11, 0x00, 0x45, 0x00, 0x00, 0x00, 0x14, 0x00, 0x50, 0x00, 0x01, 0x00, 0x46, 0x00, 0xff, 0x00, 0xf1, 0x00, 0x0a, 0xff, 0x5a, 0x0a, 0x5a, 0x00, 0x5d, 0x00, 0xff, 0x00, 0xe2, 0x00, 0x14, 0xff, 0xb4, 0x0a, 0xb4, 0x00, 0xbb, 0x00, 0xff, 0x00, 0xc4, 0x00, 0x29, 0xff, 0x69, 0x0a, 0x69, 0x00, 0x77, 0x00, 0xdd, 0x00, 0x88, 0x00, 0x52, 0xc6, 0xd2, 0x00, 0xf0 },
        { 0x00, 0x21, 0x00, 0x11, 0x00, 0x45, 0x00, 0x00, 0x00, 0x14, 0x00, 0x50, 0x00, 0x01, 0x00, 0x46, 0x00, 0xff, 0x00, 0xf1, 0x00, 0x0a, 0xff, 0x5a, 0x0a, 0x5a, 0x00, 0x5d, 0x00, 0xff, 0x00, 0xe2, 0x00, 0x14, 0xff, 0xb4, 0x0a, 0xb4, 0x00, 0xbb, 0x00, 0xff, 0x00, 0xc4, 0x00, 0x29, 0xff, 0x69, 0x0a, 0x69, 0x00, 0x77, 0x00, 0xdd, 0x00, 0x88, 0x00, 0x52, 0xc6, 0xd2, 0x00, 0xf0 },
        { 0x00, 0x21, 0x00, 0x11, 0x00, 0x45, 0x00, 0x00, 0x00, 0x14, 0x00, 0x50, 0x00, 0x01, 0x00, 0x46, 0x00, 0xff, 0x00, 0xf1, 0x00, 0x0a, 0xff, 0x5a, 0x0a, 0x5a, 0x00, 0x5d, 0x00, 0xff, 0x00, 0xe2, 0x00, 0x14, 0xff, 0xb4, 0x0a, 0xb4, 0x00, 0xbb, 0x00, 0xff, 0x00, 0xc4, 0x00, 0x29, 0xff, 0x69, 0x0a, 0x69, 0x00, 0x77, 0x00, 0xdd, 0x00, 0x88, 0x00, 0x52, 0xc6, 0xd2, 0x00, 0xf0 },
        { 0x00, 0x21, 0x00, 0x11, 0x00, 0x45, 0x00, 0x00, 0x00, 0x14, 0x00, 0x50, 0x00, 0x01, 0x00, 0x46, 0x00, 0xff, 0x00, 0xf1, 0x00, 0x0a, 0xff, 0x5a, 0x0a, 0x5a, 0x00, 0x5d, 0x00, 0xff, 0x00, 0xe2, 0x00, 0x14, 0xff, 0xb4, 0x0a, 0xb4, 0x00, 0xbb, 0x00, 0xff, 0x00, 0xc4, 0x00, 0x29, 0xff, 0x69, 0x0a, 0x69, 0x00, 0x77, 0x00, 0xdd, 0x00, 0x88, 0x00, 0x52, 0xc6, 0xd2, 0x00, 0xf0 },
    } };

std::vector<uint8_t> aac_silent_frame(int channels, int frequency_index, int element_instance_tag)
{
    // sanitise input
    frequency_index &= 0x0f;
    channels = (channels - 1) & 0x01;
    if (0 > element_instance_tag || 0x0F < element_instance_tag) {
        element_instance_tag = 0;
    }

    // unpack rle frame
    std::vector<uint8_t> silence;
    auto& packed_silence = aac_silence_frames_rle[channels][frequency_index];
    for (size_t i = 0; (i + 1) < packed_silence.size(); i += 2) {
        silence.insert(silence.end(), packed_silence[i] + 1, packed_silence[i + 1]);
    }

    // write element_instance_tag;
    silence[0] = (silence[0] & 0xE1) | ((element_instance_tag << 1) & 0x1E);
    return silence;
}

// std::vector<uint8_t> renderAdts() const
// {
//     adts_t h;
//     aac_init_adts(&h);
//     std::vector<uint8_t> data;
//     aac_parse_extradata(&h, m_extradata.data(), m_extradata.size());
//     h.frameCount = 1;
//     h.frameSize += static_cast<unsigned>(buffer.size());
//     size_t size = buffer.size();
//     data.resize(data.size() + AAC_DEFAULT_HEADER_SIZE);
//     aac_render_adts(&h, data.data() + size);
//     data.insert(data.end(), buffer.begin(), buffer.end());
//     return data;
// }
