#include "avcutil.hpp"
#include <algorithm>
#include <cmath>

class AVCBitReader {
private:
    size_t m_bpos; // bits remaining in current byte
    size_t m_ofst; // bytes remainig
    size_t m_size;
    const uint8_t* m_data;

    size_t step(size_t bits)
    {
        assert(bits <= m_bpos);
        m_bpos -= bits;
        if (0 == m_bpos) {
            ++m_ofst, m_bpos = 8;
            // We know data is at least 1, and we know the first byte
            // in a nalu is never zero. So negative indexing is bounds safe
            if (m_ofst < m_size && 3 == m_data[m_ofst] && 0 == m_data[m_ofst - 1] && 0 == m_data[m_ofst - 2]) {
                ++m_ofst; // skip emulation prevention byte
            }
        }

        return bits;
    }

    uint32_t _readExpGol()
    {
        int32_t clz = 0; // count leading zeros
        while (m_ofst < m_size && 0 == readBit()) {
            ++clz;
        }
        return (1 << clz) | readBits(clz);
    }

public:
    AVCBitReader(const uint8_t* data, size_t size, AVCParsedNalu& nalu)
        : m_bpos(8)
        , m_ofst(1) // skip first byte
        , m_size(size)
        , m_data(data)
    {
        // first byte cant be 0 or we can seg fault
        assert(0 != m_data[0]);
        if (0 == m_data[0]) {
            m_size = 0;
            nalu.forbidden_zero_bit = 1;
        } else {
            nalu.forbidden_zero_bit = (m_data[0] & 0x80) >> 7;
            nalu.nal_ref_idc = (m_data[0] & 0x60) >> 5;
            nalu.nal_unit_type = (m_data[0] & 0x1f);
        }
    }

    bool eof() const { return m_ofst >= m_size; }
    size_t positionInBits() const { return (8 * m_ofst) + (8 - m_bpos); }
    size_t positionInBytes() const { return m_ofst + (8 == m_bpos ? 0 : 1); }
    bool more_rbsp_data()
    {
        if (1 == m_size - m_ofst) {
            auto valu = 1 << (m_bpos - 1);
            auto mask = (1 << m_bpos) - 1;
            return ((m_data[m_ofst]) & mask) != valu;
        }

        return !!(m_size - m_ofst);
    }

    uint32_t rbsp_trailing_bits()
    {
        // TODO return boolean?
        uint32_t result = readBits(m_bpos);
        step(m_bpos);
        return result;
    }

    uint32_t readBit()
    {
        // slightly faster when reading a single bit
        uint32_t result = 0;
        if (m_ofst < m_size) {
            result = m_data[m_ofst] >> (m_bpos - 1);
            step(1);
        }

        return result & 1;
    }

    uint32_t readBits(size_t size)
    {
        if (1 == size) {
            // optimization for single bit
            return readBit();
        }

        uint32_t result = 0;
        while (size && m_ofst < m_size) {
            auto count = std::min(size, m_bpos);
            result = (result << count) | ((m_data[m_ofst] >> (m_bpos - count)) & ((1 << count) - 1));
            size -= step(count);
        }

        return result;
    }

    uint32_t readUExpGol() { return _readExpGol() - 1; }
    int32_t readSExpGol()
    {
        auto result = _readExpGol();
        return (result & 1 ? -1 : 1) * static_cast<int32_t>(result >> 1);
    }
};
////////////////////////////////////////////////////////////////////////////////
void parseScalingList(AVCBitReader& r, int* scalingList, int sizeOfScalingList, int* useDefaultScalingMatrixFlag)
{
    int lastScale = 8, nextScale = 8;
    for (int j = 0; j < sizeOfScalingList; j++) {
        if (nextScale != 0) {
            int delta_scale = r.readSExpGol();
            nextScale = (lastScale + delta_scale + 256) % 256;
            *useDefaultScalingMatrixFlag = (j == 0 && nextScale == 0);
        }

        scalingList[j] = (nextScale == 0) ? lastScale : nextScale;
        lastScale = scalingList[j];
    }
}

void parseRefPicListModification(class AVCBitReader& r, int slice_type,
    int* ref_pic_list_modification_flag_l0, int* ref_pic_list_modification_flag_l1,
    int* abs_diff_pic_num_minus1, int* long_term_pic_num)

{
    int modification_of_pic_nums_idc;
    if (AVCParsedVclNalu::SliceType_I != slice_type && AVCParsedVclNalu::SliceType_SI != slice_type) {
        *ref_pic_list_modification_flag_l0 = r.readBit();
        if (*ref_pic_list_modification_flag_l0) {
            do {
                modification_of_pic_nums_idc = r.readUExpGol();
                if (0 == modification_of_pic_nums_idc || 1 == modification_of_pic_nums_idc) {
                    *abs_diff_pic_num_minus1 = r.readUExpGol();
                } else if (2 == modification_of_pic_nums_idc) {
                    *long_term_pic_num = r.readUExpGol();
                }
            } while (3 != modification_of_pic_nums_idc && !r.eof());
        }
    }

    if (AVCParsedVclNalu::SliceType_B == slice_type) {
        *ref_pic_list_modification_flag_l1 = r.readBit();
        if (*ref_pic_list_modification_flag_l1) {
            do {
                modification_of_pic_nums_idc = r.readUExpGol();
                if (0 == modification_of_pic_nums_idc || 1 == modification_of_pic_nums_idc) {
                    *abs_diff_pic_num_minus1 = r.readUExpGol();
                } else if (2 == modification_of_pic_nums_idc) {
                    *long_term_pic_num = r.readUExpGol();
                }
            } while (3 != modification_of_pic_nums_idc && !r.eof());
        }
    }
}

void parsePredWeightTable(class AVCBitReader& r, int slice_type, int ChromaArrayType, int num_ref_idx_l0_active_minus1, int num_ref_idx_l1_active_minus1, AVCParsedVclNalu::PredWeightTable& pwt)
{
    pwt.luma_log2_weight_denom = r.readUExpGol();
    if (0 != ChromaArrayType) {
        pwt.chroma_log2_weight_denom = r.readUExpGol();
    }

    for (int i = 0; i <= num_ref_idx_l0_active_minus1; i++) {
        pwt.luma_weight_l0_flag = r.readBit();
        if (pwt.luma_weight_l0_flag) {
            pwt.luma_weight_l0[i] = r.readSExpGol();
            pwt.luma_offset_l0[i] = r.readSExpGol();
        }

        if (ChromaArrayType != 0) {
            pwt.chroma_weight_l0_flag = r.readBit();
            if (pwt.chroma_weight_l0_flag) {
                for (int j = 0; j < 2; j++) {
                    pwt.chroma_weight_l0[i][j] = r.readSExpGol();
                    pwt.chroma_offset_l0[i][j] = r.readSExpGol();
                }
            }
        }
    }

    if (AVCParsedVclNalu::SliceType_B == slice_type) {
        for (int i = 0; i <= num_ref_idx_l1_active_minus1; i++) {
            pwt.luma_weight_l1_flag = r.readBit();
            if (pwt.luma_weight_l1_flag) {
                pwt.luma_weight_l1[i] = r.readSExpGol();
                pwt.luma_offset_l1[i] = r.readSExpGol();
            }
            if (ChromaArrayType != 0) {
                pwt.chroma_weight_l1_flag = r.readBit();
                if (pwt.chroma_weight_l1_flag) {
                    for (int j = 0; j < 2; j++) {
                        pwt.chroma_weight_l1[i][j] = r.readSExpGol();
                        pwt.chroma_offset_l1[i][j] = r.readSExpGol();
                    }
                }
            }
        }
    }
}

void parseDecRefPicMarking(class AVCBitReader& r, int IdrPicFlag, AVCParsedVclNalu::DecRefPicMarking& drpm)
{
    if (IdrPicFlag) {
        drpm.no_output_of_prior_pics_flag = r.readBit();
        drpm.long_term_reference_flag = r.readBit();
    } else {
        int memory_management_control_operation;
        drpm.adaptive_ref_pic_marking_mode_flag = r.readBit();
        if (drpm.adaptive_ref_pic_marking_mode_flag) {
            do {
                memory_management_control_operation = r.readUExpGol();
                if (1 == memory_management_control_operation || 3 == memory_management_control_operation) {
                    drpm.difference_of_pic_nums_minus1 = r.readUExpGol();
                }

                if (2 == memory_management_control_operation) {
                    drpm.long_term_pic_num = r.readUExpGol();
                }

                if (3 == memory_management_control_operation || 6 == memory_management_control_operation) {
                    drpm.long_term_frame_idx = r.readUExpGol();
                }

                if (4 == memory_management_control_operation) {
                    drpm.max_long_term_frame_idx_plus1 = r.readUExpGol();
                }

            } while (memory_management_control_operation && !r.eof());
        }
    }
}

AVCParsedPpsNalu AVCParser::parsePps(const uint8_t* data, size_t size, const std::map<int, AVCParsedSpsNalu>& spsMap)
{
    AVCParsedPpsNalu pps;
    AVCBitReader r(data, size, pps);
    pps.pic_parameter_set_id = r.readUExpGol();
    pps.seq_parameter_set_id = r.readUExpGol();
    pps.entropy_coding_mode_flag = r.readBit();
    pps.bottom_field_pic_order_in_frame_present_flag = r.readBit();
    pps.num_slice_groups_minus1 = r.readUExpGol();

    auto spsItr = spsMap.find(pps.seq_parameter_set_id);
    const auto& sps = spsItr->second;
    if (spsMap.end() == spsItr) {
        return pps;
    }

    if (0 < pps.num_slice_groups_minus1) {
        pps.slice_group_map_type = r.readUExpGol();
        if (0 == pps.slice_group_map_type) {
            for (int iGroup = 0; iGroup <= pps.num_slice_groups_minus1; iGroup++) {
                pps.run_length_minus1[iGroup] = r.readUExpGol();
            }
        } else if (2 == pps.slice_group_map_type) {
            for (int iGroup = 0; iGroup < pps.num_slice_groups_minus1; iGroup++) {
                pps.top_left[iGroup] = r.readUExpGol();
                pps.bottom_right[iGroup] = r.readUExpGol();
            }
        } else if (3 == pps.slice_group_map_type || 4 == pps.slice_group_map_type || 5 == pps.slice_group_map_type) {
            pps.slice_group_change_direction_flag = r.readBit();
            pps.slice_group_change_rate_minus1 = r.readUExpGol();

        } else if (6 == pps.slice_group_map_type) {
            pps.pic_size_in_map_units_minus1 = r.readUExpGol();
            auto log2_num_slice_groups = std::ceil(std::log2(pps.num_slice_groups_minus1 + 1));
            for (int i = 0; i <= pps.pic_size_in_map_units_minus1; i++) {
                pps.slice_group_id[i] = r.readBits(static_cast<size_t>(log2_num_slice_groups));
            }
        }
    }

    pps.num_ref_idx_l0_default_active_minus1 = r.readUExpGol();
    pps.num_ref_idx_l1_default_active_minus1 = r.readUExpGol();
    pps.weighted_pred_flag = r.readBit();
    pps.weighted_bipred_idc = r.readBits(2);
    pps.pic_init_qp_minus26 = r.readSExpGol();
    pps.pic_init_qs_minus26 = r.readSExpGol();
    pps.chroma_qp_index_offset = r.readSExpGol();
    pps.deblocking_filter_control_present_flag = r.readBit();
    pps.constrained_intra_pred_flag = r.readBit();
    pps.redundant_pic_cnt_present_flag = r.readBit();
    if (r.more_rbsp_data()) {
        pps.transform_8x8_mode_flag = r.readBit();
        pps.pic_scaling_matrix_present_flag = r.readBit();
        if (pps.pic_scaling_matrix_present_flag) {
            for (int i = 0; i < 6 + ((sps.chroma_format_idc != 3) ? 2 : 6) * pps.transform_8x8_mode_flag; i++) {
                pps.pic_scaling_list_present_flag[i] = r.readBit();
                if (pps.pic_scaling_list_present_flag[i]) {
                    if (i < 6) {
                        parseScalingList(r, &pps.ScalingList4x4[i][0], 16, &pps.UseDefaultScalingMatrix4x4Flag[i]);
                    } else {
                        parseScalingList(r, &pps.ScalingList8x8[i - 6][0], 64, &pps.UseDefaultScalingMatrix8x8Flag[i - 6]);
                    }
                }
            }
        }
        pps.second_chroma_qp_index_offset = r.readSExpGol();
    }

    return pps;
}

// 7.3.2.1.1 Sequence parameter set data syntax
AVCParsedSpsNalu AVCParser::parseSps(const uint8_t* data, size_t size)
{
    AVCParsedSpsNalu sps;
    AVCBitReader r(data, size, sps);
    assert(NalTypeSPS == sps.nal_unit_type);
    sps.profile_idc = r.readBits(8);
    sps.constraint_set0_flag = r.readBit();
    sps.constraint_set1_flag = r.readBit();
    sps.constraint_set2_flag = r.readBit();
    sps.constraint_set3_flag = r.readBit();
    sps.constraint_set4_flag = r.readBit();
    sps.constraint_set5_flag = r.readBit();
    sps.reserved_zero_2bits = r.readBits(2);
    sps.level_idc = r.readBits(8);

    sps.seq_parameter_set_id = r.readUExpGol();

    if (sps.profile_idc == 100 || sps.profile_idc == 110 || sps.profile_idc == 122 || sps.profile_idc == 244 || sps.profile_idc == 44 || sps.profile_idc == 83 || sps.profile_idc == 86
        || sps.profile_idc == 118 || sps.profile_idc == 128 || sps.profile_idc == 138 || sps.profile_idc == 139 || sps.profile_idc == 134) {

        sps.chroma_format_idc = r.readUExpGol();

        if (3 == sps.chroma_format_idc) {
            sps.separate_colour_plane_flag = r.readBit();
        }

        sps.bit_depth_luma_minus8 = r.readUExpGol();
        sps.bit_depth_chroma_minus8 = r.readUExpGol();
        sps.qpprime_y_zero_transform_bypass_flag = r.readBit();
        sps.seq_scaling_matrix_present_flag = r.readBit();

        if (sps.seq_scaling_matrix_present_flag) {
            for (int i = 0; i < ((sps.chroma_format_idc != 3) ? 8 : 12); i++) {
                sps.seq_scaling_list_present_flag[i] = r.readBit();

                if (sps.seq_scaling_list_present_flag[i]) {
                    if (i < 6) {
                        parseScalingList(r, &sps.ScalingList4x4[i][0], 16, &sps.UseDefaultScalingMatrix4x4Flag[i]);
                    } else {
                        parseScalingList(r, &sps.ScalingList8x8[i - 6][0], 64, &sps.UseDefaultScalingMatrix8x8Flag[i - 6]);
                    }
                }
            }
        }
    }

    sps.log2_max_frame_num_minus4 = r.readUExpGol();
    sps.pic_order_cnt_type = r.readUExpGol();

    if (sps.pic_order_cnt_type == 0) {
        sps.log2_max_pic_order_cnt_lsb_minus4 = r.readUExpGol();
    } else if (sps.pic_order_cnt_type == 1) {
        sps.delta_pic_order_always_zero_flag = r.readBit();
        sps.offset_for_non_ref_pic = r.readSExpGol();
        sps.offset_for_top_to_bottom_field = r.readSExpGol();
        sps.num_ref_frames_in_pic_order_cnt_cycle = r.readUExpGol();

        for (int i = 0; i < sps.num_ref_frames_in_pic_order_cnt_cycle; i++) {
            sps.offset_for_ref_frame[i] = r.readSExpGol();
        }
    }

    sps.max_num_ref_frames = r.readUExpGol();
    sps.gaps_in_frame_num_value_allowed_flag = r.readBit();
    sps.pic_width_in_mbs_minus1 = r.readUExpGol();
    sps.pic_height_in_map_units_minus1 = r.readUExpGol();
    sps.frame_mbs_only_flag = r.readBit();

    if (!sps.frame_mbs_only_flag) {
        sps.mb_adaptive_frame_field_flag = r.readBit();
    }

    sps.direct_8x8_inference_flag = r.readBit();
    sps.frame_cropping_flag = r.readBit();

    if (sps.frame_cropping_flag) {
        sps.frame_crop_left_offset = r.readUExpGol();
        sps.frame_crop_right_offset = r.readUExpGol();
        sps.frame_crop_top_offset = r.readUExpGol();
        sps.frame_crop_bottom_offset = r.readUExpGol();
    }

    sps.vui_parameters_present_flag = r.readBit();

    if (sps.vui_parameters_present_flag) {
        sps.aspect_ratio_info_present_flag = r.readBit();

        if (sps.aspect_ratio_info_present_flag) {
            sps.aspect_ratio_idc = r.readBits(8);

            if (sps.aspect_ratio_idc == 255) {
                sps.sar_width = r.readBits(16);
                sps.sar_height = r.readBits(16);
            }
        }

        sps.overscan_info_present_flag = r.readBit();

        if (sps.overscan_info_present_flag) {
            sps.overscan_appropriate_flag = r.readBit();
        }

        sps.video_signal_type_present_flag = r.readBit();

        if (sps.video_signal_type_present_flag) {
            sps.video_format = r.readBits(3);
            sps.video_full_range_flag = r.readBit();
            sps.colour_description_present_flag = r.readBit();

            if (sps.colour_description_present_flag) {
                sps.colour_primaries = r.readBits(8);
                sps.transfer_characteristics = r.readBits(8);
                sps.matrix_coefficients = r.readBits(8);
            }
        }

        sps.chroma_loc_info_present_flag = r.readBit();

        if (sps.chroma_loc_info_present_flag) {
            sps.chroma_sample_loc_type_top_field = r.readUExpGol();
            sps.chroma_sample_loc_type_bottom_field = r.readUExpGol();
        }
    } // VUI

    return sps;
}

// TODO pass in sps/pps maps
AVCParsedVclNalu AVCParser::parseVclSliceHeader(const uint8_t* data, size_t size, const std::map<int, AVCParsedSpsNalu>& spsMap, const std::map<int, AVCParsedPpsNalu>& ppsMap)
{
    AVCParsedVclNalu nal;
    AVCBitReader r(data, size, nal);

    // TODO Nalu type type 2,3,4
    if (NalTypeSlice == nal.nal_unit_type || NalTypeIDR == nal.nal_unit_type) {
        nal.first_mb_in_slice = r.readUExpGol();
        nal.slice_type = r.readUExpGol() % 5;
        // P, B, I, SP, SI // slice types
        nal.pic_parameter_set_id = r.readUExpGol();

        auto ppsItr = ppsMap.find(nal.pic_parameter_set_id);
        const auto& pps = ppsItr->second;
        if (ppsMap.end() == ppsItr) {
            return nal;
        }

        const auto& spsItr = spsMap.find(pps.seq_parameter_set_id);
        const auto& sps = spsItr->second;
        if (spsMap.end() == spsItr) {
            return nal;
        }

        // the value of the variable ChromaArrayType is assigned as follows:
        // – If separate_colour_plane_flag is equal to 0, ChromaArrayType is set equal to chroma_format_idc.
        // – Otherwise (separate_colour_plane_flag is equal to 1), ChromaArrayType is set equal to 0.
        int ChromaArrayType = 0 == sps.separate_colour_plane_flag ? sps.chroma_format_idc : 0;
        if (1 == sps.separate_colour_plane_flag) {
            nal.colour_plane_id = r.readBits(2);
        }

        nal.frame_num = r.readBits(sps.log2_max_frame_num_minus4 + 4);
        if (!sps.frame_mbs_only_flag) {
            nal.field_pic_flag = r.readBit();
            if (nal.field_pic_flag) {
                nal.bottom_field_flag = r.readBit();
            }
        }

        if (NalTypeIDR == nal.nal_unit_type) {
            nal.idr_pic_id = r.readUExpGol();
        }

        if (0 == sps.pic_order_cnt_type) {
            nal.pic_order_cnt_lsb = r.readBits(sps.log2_max_pic_order_cnt_lsb_minus4 + 4);
            if (pps.bottom_field_pic_order_in_frame_present_flag && !nal.field_pic_flag) {
                nal.delta_pic_order_cnt_bottom = r.readSExpGol();
            }
        }

        if (1 == sps.pic_order_cnt_type && !sps.delta_pic_order_always_zero_flag) {
            nal.delta_pic_order_cnt[0] = r.readSExpGol();
            if (pps.bottom_field_pic_order_in_frame_present_flag && !nal.field_pic_flag) {
                nal.delta_pic_order_cnt[1] = r.readSExpGol();
            }
        }

        if (pps.redundant_pic_cnt_present_flag) {
            nal.redundant_pic_cnt = r.readUExpGol();
        }

        if (AVCParsedVclNalu::SliceType_B == nal.slice_type) {
            nal.direct_spatial_mv_pred_flag = r.readBit();
        }

        if (AVCParsedVclNalu::SliceType_P == nal.slice_type || AVCParsedVclNalu::SliceType_SP == nal.slice_type || AVCParsedVclNalu::SliceType_B == nal.slice_type) {
            nal.num_ref_idx_active_override_flag = r.readBit();
            if (nal.num_ref_idx_active_override_flag) {
                nal.num_ref_idx_l0_active_minus1 = r.readUExpGol();
                if (AVCParsedVclNalu::SliceType_B == nal.slice_type) {
                    nal.num_ref_idx_l1_active_minus1 = r.readUExpGol();
                }
            }
        }

        // 20 and 21 are non-VCL with a slice header. We don't care about those currently
        if (20 == nal.nal_unit_type || 21 == nal.nal_unit_type) {
            return nal;
        } else {
            parseRefPicListModification(r, nal.slice_type,
                &nal.ref_pic_list_modification_flag_l0, &nal.ref_pic_list_modification_flag_l1,
                &nal.abs_diff_pic_num_minus1, &nal.long_term_pic_num);
        }

        if ((pps.weighted_pred_flag && (AVCParsedVclNalu::SliceType_P == nal.slice_type || AVCParsedVclNalu::SliceType_SP == nal.slice_type)) || (1 == pps.weighted_bipred_idc && AVCParsedVclNalu::SliceType_B == nal.slice_type)) {
            parsePredWeightTable(r, nal.slice_type, ChromaArrayType, nal.num_ref_idx_l0_active_minus1, nal.num_ref_idx_l1_active_minus1, nal.predWeightTable);
        }

        if (0 != nal.nal_ref_idc) {
            parseDecRefPicMarking(r, (NalTypeIDR == nal.nal_unit_type), nal.decRefPicMarking);
        }

        if (pps.entropy_coding_mode_flag && AVCParsedVclNalu::SliceType_I != nal.slice_type && AVCParsedVclNalu::SliceType_SI != nal.slice_type) {
            nal.cabac_init_idc = r.readUExpGol();
        }

        nal.slice_qp_delta = r.readSExpGol();

        if (AVCParsedVclNalu::SliceType_SP == nal.slice_type || AVCParsedVclNalu::SliceType_SI == nal.slice_type) {
            if (AVCParsedVclNalu::SliceType_SP == nal.slice_type) {
                nal.sp_for_switch_flag = r.readBit();
            }
            nal.slice_qs_delta = r.readSExpGol();
        }

        if (pps.deblocking_filter_control_present_flag) {
            nal.disable_deblocking_filter_idc = r.readUExpGol();
            if (1 != nal.disable_deblocking_filter_idc) {
                nal.slice_alpha_c0_offset_div2 = r.readSExpGol();
                nal.slice_beta_offset_div2 = r.readSExpGol();
            }
        }

        if (0 < pps.num_slice_groups_minus1 && 3 <= pps.slice_group_map_type && 5 >= pps.slice_group_map_type) {
            auto picSizeInMapUnits = (sps.pic_width_in_mbs_minus1 + 1) * (sps.pic_height_in_map_units_minus1 + 1);
            auto sliceGroupChangeRate = static_cast<double>(pps.slice_group_change_rate_minus1 + 1);
            auto log2_this_mess = std::ceil(std::log2(picSizeInMapUnits / sliceGroupChangeRate + 1));
            nal.slice_group_change_cycle = r.readBits(static_cast<size_t>(log2_this_mess));
        }
    }

    nal.slice_header_size = r.positionInBytes();
    return nal;
}

AVCParsedNalu parseNaluHeader(const uint8_t* data, size_t size)
{
    AVCParsedNalu nal;
    AVCBitReader r(data, size, nal);
    return nal;
}

size_t AVCParser::parseNalu(const uint8_t* data, size_t size)
{
    assert(data && size);
    int8_t nal_unit_type = data[0] & 0x1f;
    switch (nal_unit_type) {
    default: // Not yet supported
        return 0;
        break;
    case 1:
    case 5: {
        auto nalu = parseVclSliceHeader(data, size, m_sps, m_pps);
        return static_cast<size_t>(nalu.slice_header_size);
    } break;
    case 7: {
        auto sps = parseSps(data, size);
        m_currentSpsId = sps.seq_parameter_set_id;
        m_sps.emplace(m_currentSpsId, sps);
        return 0;
    } break;
    case 8: {
        auto pps = parsePps(data, size, m_sps);
        m_currentPpsId = pps.pic_parameter_set_id;
        m_pps.emplace(m_currentPpsId, pps);
        return 0;
    } break;
    }
}
////////////////////////////////////////////////////////////////////////////////
std::pair<int, int> AVCParsedSpsNalu::resolution() const
{
    assert(0 == forbidden_zero_bit);
    int width = (pic_width_in_mbs_minus1 + 1) * 16;
    int height = (2 - frame_mbs_only_flag) * (pic_height_in_map_units_minus1 + 1) * 16;

    if (frame_cropping_flag) {
        width -= (frame_crop_left_offset + frame_crop_right_offset) * 2;
        height -= (frame_crop_top_offset + frame_crop_bottom_offset) * 2;
    }

    return std::make_pair(width, height);
}

////////////////////////////////////////////////////////////////////////////////
// only supports 3 and 4 byte start codes
// @return std::numeric_limits<size_t>::max() if too small for start code. 0 if start code. Any other value is nuber of bytes to skip before looking fo start code again
static size_t _is_start_code(const uint8_t* data, size_t size, size_t* len)
{
    if (3 > size) {
        return std::numeric_limits<size_t>::max();
    }

    if (1 < data[2]) {
        return 3;
    }

    if (0 != data[1]) {
        return 2;
    }

    if (0 == data[0]) {
        if (1 == data[2]) {
            *len = 3;
            return 0;
        }

        if (4 <= size && 1 == data[3]) {
            *len = 4;
            return 0;
        }
    }

    return 1;
}

size_t AVCParser::findStartCode(const uint8_t* data, size_t size, size_t* len)
{
    size_t pos = 0;

    for (;;) {
        // is pos pointing to a start code?
        size_t isc = _is_start_code(data + pos, size - pos, len);

        if (0 == isc) {
            // Found start code at pos;
            return pos;
        }

        if (std::numeric_limits<size_t>::max() == isc) {
            // No start code found
            return isc;
        }

        pos += isc;
    }
}

size_t AVCParser::findStartCodeIncremental(const uint8_t* data, size_t size, size_t prevSize, size_t* len)
{
    size_t offset = (3 <= prevSize) ? (prevSize - 3) : 0;
    size_t pos = findStartCode(data + offset, size - offset, len);

    if (std::numeric_limits<size_t>::max() == pos) {
        return pos;
    }

    return pos + offset;
}

AVCParser::Extradata AVCParser::parseExtradata(const std::vector<uint8_t>& extradata)
{
    AVCParser::Extradata ed;
    size_t size = extradata.size();
    const uint8_t* data = extradata.data();
    if (16 <= size) { // anything less that 16 can't possibly be legal
        ed.version = data[0];
        ed.profile = data[1];
        ed.compatibility = data[2];
        ed.level = data[3];
        ed.lengthSize = 1 + (data[4] & 0x03);
        size_t nalCount = static_cast<size_t>(data[5] & 0x1F);
        data += 6, size -= 6;

        while (nalCount && 2 <= size) {
            size_t nalSize = std::min(static_cast<size_t>((data[0] << 8) | data[1]), size - 2);
            data += 2, size -= 2;
            ed.sps.emplace_back(data, data + nalSize);
            --nalCount, data += nalSize, size -= nalSize;
        }

        if (size) {
            nalCount = *data;
            ++data, --size;
        }

        while (nalCount && 2 <= size) {
            size_t nalSize = std::min(static_cast<size_t>((data[0] << 8) | data[1]), size - 2);
            data += 2, size -= 2;
            ed.pps.emplace_back(data, data + nalSize);
            --nalCount, data += nalSize, size -= nalSize;
        }
    }

    return ed;
}

std::vector<uint8_t> AVCParser::toAnnexB(const std::vector<uint8_t>& frame, const std::vector<uint8_t>& extradata)
{
    std::vector<uint8_t> annexb;
    uint32_t nalu_mask = (1 << AVCParser::NalTypeAUD);
    size_t lengthSize = 1 + (6 <= extradata.size() ? (extradata[4] & 0x03) : 3);
    annexb.insert(annexb.end(), { 0, 0, 0, 1, 0x09, 0xf0 });

    for (auto nalu : NalIterator(frame, lengthSize)) {
        nalu_mask |= (1 << nalu.type);
        if (NalTypeIDR == nalu.type && NalBuffer::SyncSampleMask != (NalBuffer::SyncSampleMask & nalu_mask)) {
            auto ed = AVCParser::parseExtradata(extradata);
            for (auto& sps : ed.sps) {
                annexb.insert(annexb.end(), { 0, 0, 0, 1 });
                annexb.insert(annexb.end(), sps.begin(), sps.end());
            }
            for (auto& pps : ed.pps) {
                annexb.insert(annexb.end(), { 0, 0, 0, 1 });
                annexb.insert(annexb.end(), pps.begin(), pps.end());
            }
        }

        if (NalTypeAUD != nalu.type) {
            annexb.insert(annexb.end(), { 0, 0, 0, 1 });
            annexb.insert(annexb.end(), nalu.data, nalu.data + nalu.size);
        }
    }

    return annexb;
}

std::vector<uint8_t> AVCParser::getExtradataFromFrame(const std::vector<uint8_t>& frame)
{
    std::vector<uint8_t> ed;
    uint8_t spscnt = 0, ppscnt = 0;

    ed.resize(6); // save space for later
    for (auto nalu : NalIterator(frame)) {
        if (AVCParser::AVCParser::NalTypePPS == nalu.type) {
            ++ppscnt;
        } else if (AVCParser::AVCParser::NalTypeSPS == nalu.type) {
            ++spscnt;
            ed.push_back(static_cast<uint8_t>(nalu.size >> 8));
            ed.push_back(static_cast<uint8_t>(nalu.size >> 0));
            ed.insert(ed.end(), nalu.data, nalu.data + nalu.size);
        }
    }

    // If we cant construct an extradata packet, bail out.
    if (0 == ppscnt || 0 == spscnt) {
        return std::vector<uint8_t>();
    }

    ed.push_back(ppscnt);

    for (auto nalu : NalIterator(frame)) {
        if (AVCParser::AVCParser::NalTypePPS == nalu.type) {
            ed.push_back(static_cast<uint8_t>(nalu.size >> 8));
            ed.push_back(static_cast<uint8_t>(nalu.size >> 0));
            ed.insert(ed.end(), nalu.data, nalu.data + nalu.size);
        }
    }

    // populate the first 6 bytes we skipped at the begining;
    ed[0] = 0x01;
    ed[1] = ed[9];
    ed[2] = ed[10];
    ed[3] = ed[11];
    ed[4] = 0xFC | 0x03; // lengthSize = 4
    ed[5] = 0xE0 | (spscnt & 0x1F);
    return ed;
}
