#include "id3.hpp"
#include "../fourcc.hpp"
#include "debug/trace.hpp"
#include <algorithm>
#include <cstring>
#include <string>
#include <vector>

namespace twitch {
namespace media {
const int TextEncodingISO_8859_1 = 0;
//const int TextEncodingUTF16_LE = 1;
//const int TextEncodingUTF16_BE = 2;
const int TextEncodingUTF8 = 3;

// http://id3.org/id3v2.4.0-structure
const int ID3HeaderSize = 10;
const int ID3FooterSize = 10;
const int ID3FrameHeaderSize = 10;

static std::vector<std::string> id3SplitString(uint8_t encoding, const uint8_t* data, size_t size)
{
    // TODO convert from other encodings
    if (TextEncodingUTF8 != encoding && TextEncodingISO_8859_1 != encoding) {
        TRACE_DEBUG("Unsupported encoding %d", encoding);
        return std::vector<std::string>();
    }

    std::vector<std::string> fields;

    const uint8_t* end = data + size;

    while (data < end) {
        const uint8_t* delim = std::find(data, end, '\0');
        fields.push_back(std::string(data, delim));
        data = delim + 1;
    }

    return fields;
}

static uint32_t syncsafeInteger(const uint8_t* data)
{
    return ((0x7F & data[0]) << 21) | ((0x7F & data[1]) << 14) | ((0x7F & data[2]) << 7) | (0x7F & data[3]);
}

uint32_t readNullTerminatedString(const char* data, uint32_t size, std::string& str)
{
    uint32_t i = 0;
    if (data && size) {
        char c = data[i];
        while (c != '\0' && i < size) {
            str.push_back(c);
            c = data[++i];
        }
    }
    return i;
}

std::vector<std::shared_ptr<MediaSampleBuffer>> Id3::parseFrames(const std::vector<uint8_t>& buffer, MediaTime timestamp)
{
    const uint8_t* data = buffer.data();
    std::vector<std::shared_ptr<MediaSampleBuffer>> frames;
    json11::Json::array textFrames;

    if (ID3HeaderSize <= buffer.size() && 'I' == data[0] && 'D' == data[1] && '3' == data[2]) {
        //const uint16_t version = (data[3] << 8) | data[4];
        const uint8_t flags = data[5];
        int32_t size = syncsafeInteger(&data[6]);
        data += ID3HeaderSize;

        if (size + ID3HeaderSize > static_cast<int32_t>(buffer.size())) {
            TRACE_DEBUG("Truncated id3 %d > %d", size + 10, buffer.size());
            return frames;
        }

        // flags
        // a - unsync
        // b - ext header
        // c - experimental
        // d - footer present
        if (flags & 0x80) {
            TRACE_DEBUG("Unsynchronization unsupported");
            return frames;
        }

        if (flags & 0x40) {
            TRACE_DEBUG("Extended header unsupported");
            return frames;
            //int32_t extended_header_size = syncsafe_integer(&data[0]);
            //data += extended_header_size + 4; size -= extended_header_size + 4;
        }

        if (flags & 0x10) {
            size -= ID3FooterSize;
        }

        while (ID3FrameHeaderSize <= size) {
            std::string frame_id({ (char)data[0], (char)data[1], (char)data[2], (char)data[3] });
            const int32_t frame_size = syncsafeInteger(&data[4]);
            //const uint16_t frame_flags = (data[8] << 8) | data[9];
            data += ID3FrameHeaderSize;
            size -= ID3FrameHeaderSize;

            // flags format: %0h00kmnp
            // h - grouping identity flag
            // k - compression
            // m - encryption
            // n - unsynchronisation
            // p - data length indicator, it's redundant and not allowed to be set if k,m,n are unset

            if (0 >= size || frame_size > size) {
                TRACE_DEBUG("Truncated id3 %s: %d > %d", frame_id.c_str(), frame_size, size);
                break;
            }

            if (0 < frame_size) {
                // http://id3.org/id3v2.4.0-frames
                if ("TXXX" == frame_id) {
                    uint8_t encoding = data[0];
                    std::string desc;
                    auto info = id3SplitString(encoding, &data[1], frame_size - 1);

                    if (!info.empty()) {
                        // take the first string to use as the description
                        desc = info.front();
                        info.erase(info.begin());
                    }
                    textFrames.push_back(json11::Json::object {
                        { "id", frame_id },
                        { "desc", std::move(desc) },
                        { "info", std::move(info) } });

                } else if ('T' == frame_id[0]) {
                    uint8_t encoding = data[0];
                    textFrames.push_back(json11::Json::object {
                        { "id", frame_id },
                        { "info", id3SplitString(encoding, &data[1], frame_size - 1) } });

                } else if ("PRIV" == frame_id) {
                    std::string owner;
                    uint32_t offset = readNullTerminatedString(reinterpret_cast<const char*>(data), frame_size, owner);

                    auto sample = std::make_shared<MediaSampleBuffer>();
                    sample->decodeTime = timestamp;
                    sample->presentationTime = timestamp;
                    sample->isSyncSample = true;
                    // type will be set as fourcc of the frame id eg PRIV
                    sample->type = fourcc(frame_id.c_str());
                    sample->buffer.assign(data + offset + 1, data + frame_size);
                    frames.push_back(sample);
                }
            }

            data += frame_size;
            size -= frame_size;
        }

        if (0 != size) {
            TRACE_WARN("Error parsing ID3 %d bytes remaining", size);
        }
    }

    if (!textFrames.empty()) {
        json11::Json json = json11::Json::object {
            { "ID3", std::move(textFrames) }
        };
        auto sample = std::make_shared<MediaSampleBuffer>();
        sample->decodeTime = timestamp;
        sample->presentationTime = timestamp;
        sample->isSyncSample = true;
        sample->type = fourcc("json");
        std::string content;
        json.dump(content);
        sample->buffer.assign(content.begin(), content.end());
        frames.push_back(sample);
    }

    return frames;
}
}
}
