#pragma once

#include "../fourcc.hpp"
#include "playercore/MediaSample.hpp"
#include <functional>
#include <map>
#include <memory>

namespace twitch {
namespace media {
const static int64_t TSTimeScale = 90000ll;
const static int64_t TSTimestampMax = 0x1FFFFFFFFll; // 33bit
const static int64_t TSTimestampUndefined = -1ll;

const static double DefaultFrameRate = 30.0;
const static int64_t DefaultFrameDuration = static_cast<int64_t>(TSTimeScale / DefaultFrameRate);

class TransportStream;
class ElementaryStream {
public:
    ElementaryStream(uint16_t pid, uint8_t type, TransportStream* ts);
    virtual ~ElementaryStream() = default;
    // returns the number of bytes remaining in PES. 0 for AVC
    size_t remaining() const { return m_remaining; }
    void setRemaining(size_t remaining) { m_remaining = remaining; }

    enum Type : uint8_t {
        TypeAVC = 0x1B,
        TypeAAC = 0x0F,
        TypeID3 = 0x15
    };

    uint8_t type() const { return m_type; }
    uint16_t pid() const { return m_pid; }
    virtual std::vector<uint8_t> extradata() const = 0;

    void startPes(int64_t pts, int64_t dts, int data_alignment);
    bool checkContinuityCounter(int8_t continuityCounter);

    virtual void reset() = 0;
    virtual void startFrame(int64_t dts, int32_t cts, int data_alignment) = 0;
    virtual void finishFrame(int64_t dts) = 0; // dts is guaranteed to the same as the next call to startFrame()
    virtual void addData(const uint8_t* data, size_t size) = 0;
    virtual void flush() = 0;
    int64_t timestampDelta(int64_t previous, int64_t timestamp);
    // Only AVC has CTS, so we will override that there.
    virtual int64_t maxCts() const { return 0; }

    // no more than 125 FPS (125 is divisible by 90000(ts) and 1000(rtmp))
    // in the case of aac, this works out to be 128khz (1024 samples per frame), so no need to override
    // in the case of mp3 Layer II, III its 144kHz (1152 samples per frame),
    // But mp3 Layer I is 48Khz (384 samples per frame). Will probably never need support for Layer I
    virtual int64_t minDtsDelta() const { return TSTimeScale / 125; }

    // no more that 60 second gap before timeline is rewritten
    virtual int64_t maxDtsDelta() const { return TSTimeScale * 60; }

protected:
    void emitFrame(const std::shared_ptr<MediaSampleBuffer>& frame);

private:
    uint8_t m_type;
    uint16_t m_pid;
    size_t m_remaining;
    class TransportStream* m_ts;

    int64_t m_prevDts; // previous raw dts
    int64_t m_firstDts; // first raw dts
    int64_t m_currentDts; // current timestamp with rollover correction
    int8_t m_continuityCounter;
};
}
}
