#include "Mp4DrmFactory.hpp"
#include "fmp4boxes.hpp"
#include "util/Base64.hpp"
#include "util/Hex.hpp"
#include "util/Random.hpp"
#include "util/Uuid.hpp"
#include <algorithm>
#include <chrono>
#include <json11.hpp>
#include <locale>
#include <random>

#ifdef ENABLE_ENCRYPTION
#define ECB 0
#define CBC 1
#define CTR 1
#define AES128 1
#include "SampleEncrypterAES.hpp"
#include "tinyaes/aes.hpp"
#endif

namespace twitch {
namespace media {
static const size_t PerSampleIvSize = 8;

static std::array<uint8_t, 16> arrayFromVector(const std::vector<uint8_t>& v)
{
    std::array<uint8_t, 16> a { { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 } };
    std::copy_n(v.begin(), std::min(v.size(), a.size()), a.begin());
    return a;
}

std::shared_ptr<SampleEncrypter> Mp4DrmFactory::configure(uint32_t scheme, const Key_t& key, const IV_t& iv)
{
#ifdef ENABLE_ENCRYPTION
    switch (scheme) {
    default:
        return std::shared_ptr<SampleEncrypter>();
    case fourcc("cbcs"):
        return std::make_shared<SampleEncrypterAES>(SampleEncrypterAES::CBC, key, arrayFromVector(iv));
    case fourcc("cenc"):
        return std::make_shared<SampleEncrypterAES>(SampleEncrypterAES::CTR, key, arrayFromVector(iv));
    }
#endif
    (void)iv;
    (void)key;
    (void)scheme;
    // Try others here
    return std::shared_ptr<SampleEncrypter>();
}

std::vector<uint8_t> Mp4DrmFactory::generatePssh(const Uuid& systemId, const std::vector<KeyId_t>& kids, const uint8_t* data, size_t size)
{
    std::vector<uint8_t> pssh;
    pssh.resize(MoovBox::make_pssh(nullptr, 0, systemId, kids, data, size));
    pssh.resize(MoovBox::make_pssh(pssh.data(), 0, systemId, kids, data, size));
    return pssh;
}

std::vector<uint8_t> Mp4DrmFactory::generateTwitchPssh(const KeyId_t& kid)
{
    return generatePssh(ProtectionSystemClearKey, { kid }, nullptr, 0);
}

std::vector<uint8_t> Mp4DrmFactory::generateWidevinePssh(const std::vector<uint8_t>& payload)
{
    return generatePssh(ProtectionSystemWidevine, std::vector<KeyId_t>(), payload.data(), payload.size());
}

std::vector<uint8_t> Mp4DrmFactory::generatePlayReadyPssh(const std::string& payload)
{
    // Convert for utf8 to utf16. TwitchTranscoder does not have a modern compiler, so codecvt is not possible
    // The C functions use a wchar_t which has no guarantee on char size (could be utf32 on some platforms)
    // In this one case, we can assume the content will be ASCII subset from BuyDRM
    // So, we will just do a simple conversion here.
    std::string pr_header;
    pr_header.reserve(2 * payload.size());
    for (auto c : payload) {
        pr_header.push_back(c);
        pr_header.push_back(0);
    }

    return generatePssh(ProtectionSystemPlayReady, std::vector<KeyId_t>(),
        reinterpret_cast<const uint8_t*>(pr_header.data()), pr_header.size());
}

/*    {
    "created_at": "2018-03-28T13:54:57-07:00",
    "encryption_info": {
        "content_id": "e26a69aa-9f04-40e6-886e-71f8625dbc83",
        "content_key": "BchtyErdK5Wn2UW/UtMc+A==",
        "custom_xml": "<CID>qmlq4gSf5kCIbnH4Yl28gw==</CID><DRMTYPE>smooth</DRMTYPE>",
        "key_id": "1d79c67d-d4e2-4bdf-9c03-296ed1f295bf",
        "lic_acq_url": "http://sldrm.licensekeyserver.com/core/rightsmanager.asmx",
        "pr_header": "<WRMHEADER xmlns=\"http://schemas.microsoft.com/DRM/2007/03/PlayReadyHeader\" version=\"4.0.0.0\">
        <DATA><PROTECTINFO><KEYLEN>16</KEYLEN><ALGID>AESCTR</ALGID></PROTECTINFO>
        <KID>fcZ5HeLU30ucAylu0fKVvw==</KID>
        <LA_URL>http://sldrm.licensekeyserver.com/core/rightsmanager.asmx</LA_URL>
        <DS_ID>VlR7IdsIJEuRd06Laqs2jw==</DS_ID><CUSTOMATTRIBUTES xmlns=\"\">
        <CID>qmlq4gSf5kCIbnH4Yl28gw==</CID><DRMTYPE>smooth</DRMTYPE></CUSTOMATTRIBUTES><CHECKSUM>cNg5+4t20vs=</CHECKSUM></DATA></WRMHEADER>",
        "service_id": "217b5456-08db-4b24-9177-4e8b6aab368f",
        "status": "0",
        "wv_header": "CAESEB15xn3U4kvfnAMpbtHylb8aC2J1eWRybWtleW9zIhDiammqnwRA5ohucfhiXbyDKgJIRA=="
    },
    "stream_id": "00000000-0000-0000-0000-000000000099"
}
*/

std::vector<EncryptionInfo> Mp4DrmFactory::parseSkelligeJson(const std::string& data, const std::string& scheme_type)
{
    std::string err;
    // TODO check status code
    std::vector<EncryptionInfo> infos;
    json11::Json resp = json11::Json::parse(data, err);

    for (auto& item : resp.array_items()) {
        EncryptionInfo info;

        auto key = twitch::Base64::decode(item["content_key"].string_value());
        auto key_id = twitch::Uuid::fromString(item["key_id"].string_value()).toBytes();
        auto pr_header = item["pr_header"].string_value();
        auto wv_header = twitch::Base64::decode(item["wv_header"].string_value());

        if (!wv_header.empty()) {
            info.pssh.push_back(generateWidevinePssh(wv_header));
        }

        if (!pr_header.empty()) {
            info.pssh.push_back(generatePlayReadyPssh(pr_header));
        }

        info.scheme = fourcc(scheme_type.c_str());
        info.key = arrayFromVector(key);
        info.kid = arrayFromVector(key_id);
        info.lic_acq_url = item["lic_acq_url"].string_value();
        if (fourcc("cbcs") == info.scheme) {
            info.iv = IV_t(info.kid.begin(), info.kid.end());
        } else {
            info.iv = twitch::Random::buffer(PerSampleIvSize);
        }

        infos.push_back(info);
    }

    return infos;
}

std::vector<EncryptionInfo> Mp4DrmFactory::clearKeyTestInfo()
{
    std::vector<EncryptionInfo> ret;
    for (int i = 0; i < 1; ++i) {
        EncryptionInfo info;
        info.iv = twitch::Random::buffer(PerSampleIvSize);
        info.kid = arrayFromVector(info.iv);
        info.key = info.kid;
        info.scheme = fourcc("cenc");
        info.pssh.push_back(generateTwitchPssh(info.kid));
        ret.push_back(info);
    }
    return ret;
}
}
}
