#pragma once

#include "Mp4Track.hpp"
#include "media/Stream.hpp"
#include "mp4types.hpp"
#include "playercore/Log.hpp"
#include "playercore/MediaResult.hpp"
#include "playercore/MediaSample.hpp"
#include "playercore/MediaTime.hpp"
#include <functional>
#include <map>
#include <memory>
#include <string>

namespace twitch {
namespace media {
class Mp4Parser {
public:
    explicit Mp4Parser(std::shared_ptr<Log> log);
    virtual ~Mp4Parser() = default;

    using TrackList = std::vector<std::shared_ptr<Mp4Track>>;
    using SampleCallback = std::function<void(const Mp4Track& track, std::shared_ptr<MediaSampleBuffer> sample)>;

    void setStream(Stream* stream);
    bool canReadTracks();
    bool canReadSamples(const TrackList& tracks, MediaTime duration = MediaTime::max());

    MediaResult readTracks();
    MediaResult readBoxes(const mp4box& parent, std::function<bool(mp4box&)>&& callback);
    MediaResult readBoxes(size_t start, size_t limit, std::function<bool(mp4box&)>&& callback);
    MediaResult readSamples(const TrackList& tracks, const SampleCallback& callback, MediaTime maxDuration = MediaTime::max());
    MediaResult seekTo(MediaTime time);
    bool isEnded() const;
    const std::vector<std::shared_ptr<Mp4Track>>& getTracks() const { return m_tracks; }
    bool isInitializationFragment() const { return m_isFragmented && m_isInitFragment; }
    bool isFragmented() const { return m_isFragmented; }
    size_t getNextFragmentOffset() const { return m_nextFragmentOffset; }
    int getSequenceNumber() const { return m_sequenceNumber; }
    std::shared_ptr<Mp4Track> getTrackById(uint32_t id);
    static uint32_t getSampleDuration(const Mp4Track& track, const mp4sample& sample);
    MediaTime getDuration() const;
    const std::vector<mp4emsg>& getEmsgs() { return m_emsgs; }
    const std::vector<std::vector<uint8_t>>& getPsshBytes() const { return m_psshBytes; }
    std::vector<uint8_t> getSampleKeyId(const Mp4Track& track, size_t sampleNo);
    std::vector<uint8_t> getSampleIV(const Mp4Track& track, size_t sampleNo);

protected:
    virtual void unhandledBox(const mp4box& parent, mp4box& box);

private:
    size_t getTrackOffset(const Mp4Track& track) const;
    uint8_t readUint8();
    uint16_t readUint16();
    uint32_t readUint32();
    uint64_t readUint64();
    std::vector<uint8_t> readBuffer(size_t size);
    std::string readNullTerminatedString(size_t maxSize);
    void readCodecData(Mp4Track& track, const mp4box& box, size_t read);

    void readBox(mp4box& box);
    void read_emsg(const mp4box& emsg);
    void read_ftyp();
    void read_moov(const mp4box& moov);
    void read_moof(const mp4box& moof);
    void read_mvex(const mp4box& mvex);
    void read_mvhd();
    void read_mfhd();
    void read_pssh();
    void read_trak(const mp4box& trak);
    void read_traf(const mp4box& traf);
    void read_trex();
    void read_mdia(Mp4Track& track, const mp4box& mdia);
    void read_mdhd(Mp4Track& track);
    void read_minf(Mp4Track& track, const mp4box& minf);
    void read_hdlr(Mp4Track& track, const mp4box& hdlr);
    void read_stbl(Mp4Track& track, const mp4box& stbl);
    void read_stco(Mp4Track& track);
    void read_co64(Mp4Track& track);
    void read_stsc(Mp4Track& track);
    void read_stsd(Mp4Track& track, const mp4box& stsd);
    void read_stss(Mp4Track& track);
    void read_ctts(Mp4Track& track);
    void read_stts(Mp4Track& track);
    void read_stsz(Mp4Track& track);
    void read_tfdt(Mp4Track& track);
    void read_tkhd(Mp4Track& track);
    void read_trun(Mp4Track& track);
    // DRM BOXES:
    void read_sbgp(Mp4Track& track);
    void read_sgpd(Mp4Track& track);
    void read_saiz(Mp4Track& track);
    void read_saio(Mp4Track& track);
    void read_senc(Mp4Track& track);
    void read_sinf(Mp4Track& track, const mp4box& sinf);
    void read_schi(Mp4Track& track, const mp4box& schi);

    std::shared_ptr<Mp4Track> read_tfhd();

    std::shared_ptr<Log> m_log;
    TrackList m_tracks;
    std::map<uint32_t, mp4tfhd> m_trackFragmentDefaults;
    mp4ftyp m_ftyp;
    mp4mvhd m_mvhd;
    mp4box m_mdat;
    size_t m_moofOffset;
    size_t m_nextFragmentOffset;
    bool m_isInitFragment;
    bool m_isFragmented;
    uint32_t m_sequenceNumber;
    Stream* m_stream;
    std::vector<mp4pssh> m_psshBoxes;
    std::vector<std::vector<uint8_t>> m_psshBytes;
    std::vector<mp4emsg> m_emsgs;
};
}
}
