#include "Mp4Track.hpp"

namespace twitch {
namespace media {
Mp4Track::Mp4Track()
    : m_head {}
    , m_header {}
    , m_fragment {}
    , m_run {}
    , m_baseMediaDecodeTime(0)
    , m_duration(0)
    , m_codecBoxType(0)
    , m_handlerType(0)
    , m_sampledIndex(0)
    , m_sampledDuration(0)
    , m_elstMediaTime(0)
    , m_sencOffset(0)
    , m_originalFormat(0)
    , m_protectionSchemeType(0)
    , m_protectionSchemeVersion(0)
    , m_patternEncryption(0)
    , m_defaultIsProtected(1)
    , m_defaultPerSampleIvSize(16)
    , m_currentEncryptionIndex(0)
{
}

Mp4Track::Mp4Track(uint32_t trackId,
    uint32_t format,
    uint32_t timescale,
    uint32_t handlerType,
    uint32_t defaultSampleSize,
    uint32_t defaultSampleFlags,
    uint32_t defaultSampleDuration,
    int32_t elstMediaTime,
    const std::string& name,
    const std::vector<uint8_t>& extradata,
    const std::vector<EncryptionInfo>& encryptionInfo)
{
    setEncryptionInfo(encryptionInfo);
    m_duration = 0;
    m_handlerType = handlerType;
    m_handlerName = name;
    m_codecData = extradata;
    m_baseMediaDecodeTime = 0;
    m_elstMediaTime = elstMediaTime;
    m_run.default_sample_size = defaultSampleSize;
    m_run.default_sample_flags = defaultSampleFlags;
    m_run.default_sample_duration = defaultSampleDuration;
    m_header.timescale = timescale;
    m_head.track_ID = trackId;
    m_originalFormat = format;
}

bool Mp4Track::isProtected() const
{
    return m_codecBoxType == MP4_encv || m_codecBoxType == MP4_enca || !m_encryptionInfo.empty();
}

MediaTime Mp4Track::scaleTime(uint64_t time) const
{
    return MediaTime(time, getTimescale());
}

uint64_t Mp4Track::scaleTime(MediaTime time) const
{
    return static_cast<uint64_t>(time.scaleTo(getTimescale()).count());
}

void Mp4Track::clearSamples()
{
    m_data.clear();
    m_samples.clear();
    m_baseMediaDecodeTime += m_duration;
    m_duration = 0;
}

void Mp4Track::addSample(const MediaSampleBuffer& sample)
{
    int64_t dts = sample.decodeTime.scaleTo(getTimescale()).count();
    int32_t cts = 0;
    uint32_t duration = static_cast<uint32_t>(sample.duration.scaleTo(getTimescale()).count());
    uint32_t flags = 0;
    if (m_handlerType == MP4_soun) {
        flags = SampleFlagsAudio;
    } else if (m_handlerType == MP4_vide) {
        flags = sample.isSyncSample ? SampleFlagsIntra : SampleFlagsPredicted;
        cts = static_cast<int32_t>((sample.presentationTime - sample.decodeTime).scaleTo(getTimescale()).count());
    }
    const uint8_t* data = sample.buffer.data();
    uint32_t size = static_cast<uint32_t>(sample.buffer.size());
    addSample(dts, cts, duration, flags, data, size);
}

bool Mp4Track::addSample(const mp4sample& sample, const uint8_t* data)
{
    if (m_samples.empty()) {
        m_duration = 0;
        m_baseMediaDecodeTime = static_cast<uint64_t>(sample.decodeTime);
    } else {
        // assert(m_samples.back().dts + m_samples.back().duration == sample.dts); // Frame time Misalignment
    }

    m_samples.push_back(sample);
    m_data.insert(m_data.end(), data, data + sample.size);
    m_duration += sample.duration;
    return true;
}

bool Mp4Track::addSample(int64_t dts, int32_t cts, uint32_t duration, uint32_t flags, const uint8_t* data, uint32_t size)
{
    mp4sample sample {};
    sample.decodeTime = dts;
    sample.compositionTimeOffset = cts;
    sample.duration = duration;
    sample.flags = flags;
    sample.size = size;
    return addSample(sample, data);
}

void Mp4Track::setEncryptionInfo(const std::vector<EncryptionInfo>& encryptionInfo)
{
    m_currentEncryptionIndex = 0;
    m_encryptionInfo = encryptionInfo;
}

const EncryptionInfo& Mp4Track::getCurrentEncryptionInfo() const
{
    if (m_encryptionInfo.size() <= m_currentEncryptionIndex) {
        static const EncryptionInfo info;
        return info;
    }

    return m_encryptionInfo[m_currentEncryptionIndex];
}

void Mp4Track::rotateEncryptionInfo()
{
    // Key rotation not currently supported
    // if (1 < m_encryptionInfo.size()) {
    //     m_currentEncryptionIndex = (1 + m_currentEncryptionIndex) % m_encryptionInfo.size();
    // }
}
}
}
