#pragma once

#include "EncryptionInfo.hpp"
#include "Mp4DrmFactory.hpp"
#include "mp4sample.hpp"
#include "mp4types.hpp"
#include "playercore/MediaSample.hpp"
#include "playercore/MediaTime.hpp"
#include <memory>
#include <string>
#include <vector>

namespace twitch {
namespace media {
class Mp4Track {
public:
    Mp4Track();

    Mp4Track(uint32_t trackId,
        uint32_t format,
        uint32_t timescale,
        uint32_t handlerType,
        uint32_t defaultSampleSize,
        uint32_t defaultSampleFlags,
        uint32_t defaultSampleDuration,
        int32_t elstMediaTime,
        const std::string& name,
        const std::vector<uint8_t>& extradata,
        const std::vector<EncryptionInfo>& encryptionInfo);

    virtual ~Mp4Track() = default;
    uint32_t getId() const { return m_head.track_ID; }
    const std::string& getHandlerName() const { return m_handlerName; }
    uint32_t getHandlerType() const { return m_handlerType; }
    uint32_t getCodecBoxType() const { return m_codecBoxType; }
    uint32_t getOriginalFormat() const { return m_originalFormat; }
    uint32_t getTimescale() const { return m_header.timescale; }
    const std::vector<uint8_t>& getCodecData() const { return m_codecData; }
    const mp4audio& getAudioInfo() const { return m_audio; }
    const mp4visual& getVideoInfo() const { return m_video; }
    uint64_t getBaseMediaDecodeTime() const { return m_baseMediaDecodeTime; }
    uint64_t getDuration() const { return m_duration; }
    MediaTime getDurationTime() const { return scaleTime(m_duration); }
    int32_t getElstMediaTime() const { return m_elstMediaTime; }
    void setElstMediatime(int32_t time) { m_elstMediaTime = time; }

    bool isProtected() const;
    MediaTime scaleTime(uint64_t time) const;
    uint64_t scaleTime(MediaTime time) const;
    const std::vector<mp4sample>& getSamples() const { return m_samples; }
    // fmp4
    const mp4trun& getTrackRun() const { return m_run; }

    void rotateEncryptionInfo();
    size_t getCurrentEncryptionIndex() const { return m_currentEncryptionIndex; }
    bool useSubSampleEncryption() const { return MP4_avc1 == m_originalFormat; }
    const EncryptionInfo& getCurrentEncryptionInfo() const;
    void setEncryptionInfo(const std::vector<EncryptionInfo>& encryptionInfo);
    const std::vector<EncryptionInfo>& getEncryptionInfo() const { return m_encryptionInfo; }

    static constexpr uint32_t SampleFlagsAudio
        = (mp4sample::SampleDependsOnOthersNo | mp4sample::SampleIsDependedOnNo);

    static constexpr uint32_t SampleFlagsText
        = (mp4sample::SampleDependsOnOthersNo
            | mp4sample::SampleIsDependedOnNo);

    static constexpr uint32_t SampleFlagsIntra
        = (mp4sample::SampleDependsOnOthersNo
            | mp4sample::SampleIsDependedOnYes);

    static constexpr uint32_t SampleFlagsPredicted
        = (mp4sample::SampleDependsOnOthersYes
            | mp4sample::SampleIsDependedOnYes
            | mp4sample::SampleIsNonSync);

    static constexpr uint32_t SampleFlagsBiDirectional
        = (mp4sample::SampleDependsOnOthersYes
            | mp4sample::SampleIsDependedOnYes
            | mp4sample::SampleIsNonSync
            | mp4sample::SampleIsLeadingDecodable);

    void clearSamples();
    void addSample(const MediaSampleBuffer& sample);
    virtual bool addSample(int64_t dts, int32_t cts, uint32_t duration, uint32_t flags, const uint8_t* data, uint32_t size);
    const std::vector<uint8_t>& data() const { return m_data; }

protected:
    bool addSample(const mp4sample& sample, const uint8_t* data);

private:
    std::vector<uint8_t> m_data;

protected:
    friend class Mp4Parser;
    mp4tkhd m_head;
    mp4mdhd m_header;
    mp4tfhd m_fragment;
    mp4trun m_run;
    std::vector<mp4sample> m_samples;
    uint64_t m_baseMediaDecodeTime;
    uint64_t m_duration;
    std::vector<uint32_t> m_syncSamples;
    mp4audio m_audio;
    mp4visual m_video;
    uint32_t m_codecBoxType;
    uint32_t m_handlerType;
    std::string m_handlerName;
    std::vector<uint8_t> m_codecData;
    std::vector<mp4chunk> m_chunks;
    uint32_t m_sampledIndex;
    uint64_t m_sampledDuration;
    int32_t m_elstMediaTime = 0;

    // DRM
    struct SampleGroup {
        uint8_t isProtected = 0;
        uint8_t perSampleIVSize = 0;
        uint8_t patternEncryption = 0;
        std::vector<uint8_t> constantIV;
        std::vector<uint8_t> kid;
    };

    size_t m_sencOffset = 0;
    uint32_t m_originalFormat = 0; // avc1/mp4a/etc
    uint32_t m_protectionSchemeType = 0; // cenc/cbcs
    uint32_t m_protectionSchemeVersion = 0;
    uint8_t m_patternEncryption = 0;
    uint8_t m_defaultIsProtected = 1;
    uint8_t m_defaultPerSampleIvSize = 16;
    std::vector<uint8_t> m_defaultKeyId;
    std::vector<uint8_t> m_defaultConstantIv;
    std::vector<uint32_t> m_sampleToGroup;
    std::vector<SampleGroup> m_sampleGroup;

    size_t m_currentEncryptionIndex = 0;
    std::vector<EncryptionInfo> m_encryptionInfo;
};
}
}
