#include "fmp4.hpp"
#include "../fourcc.hpp"
#include "fmp4boxes.hpp"
#include <algorithm>

namespace twitch {
namespace media {
fmp4::fmp4(uint32_t sequenceNumber)
    : m_needInit(true)
    , m_nextTrackId(AudioTrackId)
    , m_sequenceNumber(sequenceNumber)
{
}

int64_t fmp4::fragmentDuration(uint32_t timebase) const
{
    timebase = timebase ? timebase : MOOV_TIMESCALE;
    const auto audioTrack = m_tracks.find(AudioTrackId);
    if (audioTrack != m_tracks.end() && !!audioTrack->second) {
        // convert from track timebase to media timebase
        MediaTime duration = audioTrack->second->getDurationTime();
        return duration.scaleTo(timebase).count();
    }

    return 0;
}

int64_t fmp4::latestPresentationTime(uint32_t timebase) const
{
    int64_t lpt = std::numeric_limits<int64_t>::max();

    for (const auto& track : m_tracks) {
        MediaTime end(track.second->getBaseMediaDecodeTime() + track.second->getDuration(), track.second->getTimescale());
        lpt = std::min(lpt, end.scaleTo(timebase).count());
    }
    return lpt;
}

int64_t fmp4::earliestPresentationTime(uint32_t timebase) const
{
    int64_t ept = std::numeric_limits<int64_t>::max();

    for (const auto& track : m_tracks) {
        MediaTime baseMediaDecode(track.second->getBaseMediaDecodeTime(), track.second->getTimescale());
        ept = std::min(ept, baseMediaDecode.scaleTo(timebase).count());
    }
    return ept;
}

bool fmp4::samplesPending()
{
    for (const auto& track : tracks()) {
        if (!track.second->getSamples().size()) {
            return false;
        }
    }

    // return false if there are no tracks
    return !!tracks().size();
}

std::shared_ptr<fmp4track_avc> fmp4::addTrackAvc(uint32_t timescale,
    uint32_t defaultSampleSize, uint32_t defaultSampleFlags, uint32_t defaultSampleDuration,
    int32_t elstMediaTime, const std::vector<uint8_t>& extradata,
    const std::vector<EncryptionInfo>& encryptionInfo)
{
    auto track = std::make_shared<fmp4track_avc>(VideoTrackId, timescale,
        defaultSampleSize, defaultSampleFlags, defaultSampleDuration,
        elstMediaTime, extradata, encryptionInfo);

    m_nextTrackId = std::max(m_nextTrackId, VideoTrackId + 1);
    m_tracks[VideoTrackId] = track;

    return track;
}

std::shared_ptr<fmp4track_aac> fmp4::addTrackAac(uint32_t timescale,
    uint32_t defaultSampleSize, uint32_t defaultSampleFlags, uint32_t defaultSampleDuration,
    int32_t elstMediaTime, const std::vector<uint8_t>& extradata,
    const std::vector<EncryptionInfo>& encryptionInfo)
{
    auto track = std::make_shared<fmp4track_aac>(AudioTrackId, timescale,
        defaultSampleSize, defaultSampleFlags, defaultSampleDuration,
        elstMediaTime, extradata, encryptionInfo);

    m_nextTrackId = std::max(m_nextTrackId, AudioTrackId + 1);
    m_tracks[AudioTrackId] = track;
    return track;
}

std::shared_ptr<fmp4track_vp9> fmp4::addTrackVP9(uint32_t timescale,
    uint32_t defaultSampleSize, uint32_t defaultSampleFlags, uint32_t defaultSampleDuration,
    int32_t elstMediaTime,
    const fmp4track_vp9::VP9ConfigurationRecord& configurationRecord)
{
    auto track = std::make_shared<fmp4track_vp9>(VideoTrackId, timescale,
        defaultSampleSize, defaultSampleFlags, defaultSampleDuration,
        elstMediaTime, configurationRecord);

    m_nextTrackId = std::max(m_nextTrackId, VideoTrackId + 1);
    m_tracks[VideoTrackId] = track;

    return track;
}

std::shared_ptr<fmp4track_webvtt> fmp4::addTrackWebVTT(uint32_t timescale,
    uint32_t defaultSampleSize, uint32_t defaultSampleFlags, uint32_t defaultSampleDuration,
    int32_t elstMediaTime)
{
    auto track = std::make_shared<fmp4track_webvtt>(TextTrackId, timescale,
        defaultSampleSize, defaultSampleFlags, defaultSampleDuration,
        elstMediaTime);

    m_nextTrackId = std::max(m_nextTrackId, TextTrackId + 1);
    m_tracks[TextTrackId] = track;

    return track;
}

std::shared_ptr<fmp4track_opus> fmp4::addTrackOpus(uint32_t timescale,
    uint32_t defaultSampleSize, uint32_t defaultSampleFlags, uint32_t defaultSampleDuration,
    int32_t elstMediaTime, const fmp4track_opus::OpusConfiguration& opusConfig)
{
    auto track = std::make_shared<fmp4track_opus>(AudioTrackId, timescale,
        defaultSampleSize, defaultSampleFlags, defaultSampleDuration,
        elstMediaTime, opusConfig);

    m_nextTrackId = std::max(m_nextTrackId, AudioTrackId + 1);
    m_tracks[TextTrackId] = track;

    return track;
}

void fmp4::addEMsg(const std::string& scheme_id_uri,
    const std::string& value,
    uint32_t id,
    uint32_t event_duration,
    int64_t presentation_time,
    const std::vector<uint8_t>& data)
{
    addEMsg({ scheme_id_uri, value, MOOV_TIMESCALE, static_cast<uint32_t>(presentation_time), event_duration, id, data });
}

static size_t make_initialization(uint8_t* data, const fmp4& mp4)
{
    size_t i = 0;
    i += MoovBox::make_ftyp(data ? (data + i) : nullptr, fourcc("mp42"), 1,
        { fourcc("isom"), fourcc("mp42"), fourcc("dash"), fourcc("avc1"), fourcc("iso6"),
            fourcc("hlsf") });
    i += MoovBox::make_moov(data ? (data + i) : nullptr, mp4);
    return i;
}

bool fmp4::renderMoov(std::vector<uint8_t>& buffer)
{
    size_t size = buffer.size();
    uint32_t init_size = static_cast<uint32_t>(make_initialization(0, *this));
    buffer.resize(size + init_size);
    make_initialization(buffer.data() + size, *this);
    m_needInit = false;
    return true;
}

bool fmp4::renderMoof(std::vector<uint8_t>& buffer)
{
    if (!samplesPending()) {
        return false;
    }

    // Sort and render emsh for this fragment. Save future emsg for later
    std::sort(m_emsg.begin(), m_emsg.end(), [](const mp4emsg& a, const mp4emsg& b) {
        return a.presentation_time_delta < b.presentation_time_delta;
    });

    std::vector<mp4emsg> emsg_temp;
    int64_t latest_presentation_time = latestPresentationTime(MOOV_TIMESCALE);
    int64_t earliest_presentation_time = earliestPresentationTime(MOOV_TIMESCALE);

    for (const auto& emsg : m_emsg) {
        if (emsg.presentation_time_delta <= latest_presentation_time) {
            size_t size = buffer.size();
            uint32_t presentation_time_delta = emsg.presentation_time_delta <= earliest_presentation_time
                ? 0
                : static_cast<uint32_t>(emsg.presentation_time_delta - earliest_presentation_time);
            auto emsg_size = MoovBox::make_emsg(nullptr, emsg.scheme_id_uri, emsg.value, MOOV_TIMESCALE,
                presentation_time_delta, emsg.event_duration, emsg.id, emsg.data);
            buffer.resize(size + emsg_size);
            MoovBox::make_emsg(buffer.data() + size, emsg.scheme_id_uri, emsg.value, MOOV_TIMESCALE,
                presentation_time_delta, emsg.event_duration, emsg.id, emsg.data);
        } else {
            // Save for later
            emsg_temp.push_back(emsg);
        }
    }

    m_emsg.swap(emsg_temp);
    size_t size = buffer.size(); // Get size after emesg are rendered

    // TODO can we put an avc3 in the moof and change renditions without a new init?
    uint32_t moof_size = static_cast<uint32_t>(MoovBox::make_moof(nullptr, 0, 0, *this));
    uint32_t mdat_size = BOX_HEADER_SIZE;

    for (const auto& track : tracks()) {
        mdat_size += static_cast<uint32_t>(track.second->data().size());
    }

    // prepare buffer and make moof
    buffer.reserve(size + moof_size + mdat_size);
    buffer.resize(size + moof_size + BOX_HEADER_SIZE);
    size += MoovBox::make_moof(buffer.data() + size, m_sequenceNumber, moof_size + BOX_HEADER_SIZE, *this);

    // Now append the mdat
    MoovBox::make_mdat(buffer.data() + size, mdat_size);

    for (const auto& track : tracks()) {
        buffer.insert(buffer.end(), track.second->data().begin(), track.second->data().end());
        track.second->clearSamples();
    }

    ++m_sequenceNumber;
    return !buffer.empty();
}

std::vector<uint8_t> fmp4::renderSkip() const
{
    return MoovBox::make_skip();
}

std::vector<uint8_t> fmp4::renderWebVTT(const std::string& payload) const
{
    return MoovBox::make_vttc(payload);
}

std::vector<uint8_t> fmp4::renderEmptyWebVTT() const
{
    return MoovBox::make_vtte();
}

bool fmp4::renderTo(std::vector<uint8_t>& buffer)
{
    if (m_needInit) {
        renderMoov(buffer);
    }
    renderMoof(buffer);
    return !buffer.empty();
}
}
}
