#pragma once

#include "fmp4track.hpp"
#include "mp4types.hpp"
#include <memory>

#define MOOV_TIMESCALE 100000

namespace twitch {
namespace media {
class fmp4 {
public:
    explicit fmp4(uint32_t sequenceNumber = 1);
    uint32_t sequenceNumber() const { return m_sequenceNumber; }

    bool needInit() const { return m_needInit; }
    bool renderTo(std::vector<uint8_t>& buffer);
    bool renderMoov(std::vector<uint8_t>& buffer);
    bool renderMoof(std::vector<uint8_t>& buffer);
    std::vector<uint8_t> renderSkip() const;
    std::vector<uint8_t> renderWebVTT(const std::string& payload) const;
    std::vector<uint8_t> renderEmptyWebVTT() const;
    bool samplesPending();

    int64_t fragmentDuration(uint32_t timebase) const;
    int64_t latestPresentationTime(uint32_t timebase) const;
    int64_t earliestPresentationTime(uint32_t timebase) const;

    // presentation_time and event_duration MUST be in MOOV_TIMESCALE timebase
    void addEMsg(const mp4emsg& data) { m_emsg.push_back(data); }
    void addEMsg(const std::string& scheme_id_uri,
        const std::string& value,
        uint32_t id,
        uint32_t event_duration,
        int64_t presentation_time,
        const std::vector<uint8_t>& data);

    std::shared_ptr<fmp4track_avc> addTrackAvc(uint32_t timescale,
        uint32_t defaultSampleSize, uint32_t defaultSampleFlags, uint32_t defaultSampleDuration,
        int32_t elstMediaTime,
        const std::vector<uint8_t>& extradata,
        const std::vector<EncryptionInfo>& encryptionInfo);

    std::shared_ptr<fmp4track_aac> addTrackAac(uint32_t timescale,
        uint32_t defaultSampleSize, uint32_t defaultSampleFlags, uint32_t defaultSampleDuration,
        int32_t elstMediaTime,
        const std::vector<uint8_t>& extradata,
        const std::vector<EncryptionInfo>& encryptionInfo);

    std::shared_ptr<fmp4track_vp9> addTrackVP9(uint32_t timescale,
        uint32_t defaultSampleSize, uint32_t defaultSampleFlags, uint32_t defaultSampleDuration,
        int32_t elstMediaTime,
        const fmp4track_vp9::VP9ConfigurationRecord& configurationRecord);

    std::shared_ptr<fmp4track_webvtt> addTrackWebVTT(uint32_t timescale,
        uint32_t defaultSampleSize, uint32_t defaultSampleFlags, uint32_t defaultSampleDuration,
        int32_t elstMediaTime);

    std::shared_ptr<fmp4track_opus> addTrackOpus(uint32_t timescale,
        uint32_t defaultSampleSize, uint32_t defaultSampleFlags, uint32_t defaultSampleDuration,
        int32_t elstMediaTime,
        const fmp4track_opus::OpusConfiguration& opusConfig);

    std::shared_ptr<fmp4track_avc> addTrackAvc(uint32_t timescale,
        const std::vector<uint8_t>& extradata,
        const std::vector<EncryptionInfo>& encryptionInfo)
    {
        return addTrackAvc(timescale, 0, Mp4Track::SampleFlagsIntra, (timescale / 30), 0, extradata, encryptionInfo);
    }

    std::shared_ptr<fmp4track_aac> addTrackAac(uint32_t timescale,
        const std::vector<uint8_t>& extradata,
        const std::vector<EncryptionInfo>& encryptionInfo)
    {
        return addTrackAac(timescale, 0, Mp4Track::SampleFlagsAudio, (timescale / 1024), 0, extradata, encryptionInfo);
    }

    std::shared_ptr<fmp4track_vp9> addTrackVP9(uint32_t timescale, const fmp4track_vp9::VP9ConfigurationRecord& configurationRecord)
    {
        return addTrackVP9(timescale, 0, Mp4Track::SampleFlagsIntra, (timescale / 30), 0, configurationRecord);
    }

    std::shared_ptr<fmp4track_webvtt> addTrackWebVTT(uint32_t timescale)
    {
        return addTrackWebVTT(timescale, 0, Mp4Track::SampleFlagsText, (timescale / 30), 0);
    }

    std::shared_ptr<fmp4track_opus> addTrackOpus(uint32_t timescale, const fmp4track_opus::OpusConfiguration& opusConfig)
    {
        return addTrackOpus(timescale, 0, Mp4Track::SampleFlagsText, (timescale / 960), 0, opusConfig);
    }

    const std::map<uint32_t, std::shared_ptr<Mp4Track>>& tracks() const { return m_tracks; }
    std::shared_ptr<Mp4Track> track(int id) { return m_tracks.at(id); }
    int nextTrackId() const { return m_nextTrackId; }

private:
    // Hardcoding trackId is a hack.
    // But some version of safari can not handle if a trackId changes between initialization fragments
    enum {
        AudioTrackId = 1,
        VideoTrackId = 2,
        TextTrackId = 3,
    };

    bool m_needInit;
    int m_nextTrackId;
    uint32_t m_sequenceNumber;
    std::vector<mp4emsg> m_emsg;
    std::map<uint32_t, std::shared_ptr<Mp4Track>> m_tracks;
};
}
}
