#include "AsyncHttpClient.hpp"
#include "AsyncHttpRequest.hpp"
#include "AsyncHttpResponse.hpp"

namespace twitch {
AsyncHttpClient::AsyncHttpClient(Platform& platform, std::shared_ptr<Scheduler> scheduler)
    : m_httpClient(platform.getHttpClient())
    , m_ioScheduler(platform.createScheduler("Player AsyncHttp"))
    , m_targetScheduler(std::move(scheduler))
{
}

std::shared_ptr<HttpRequest> AsyncHttpClient::createRequest(const std::string& url, HttpMethod method)
{
    return std::make_shared<AsyncHttpRequest>(url, method, m_httpClient->createRequest(url, method));
}

void AsyncHttpClient::send(std::shared_ptr<HttpRequest> request, ResponseHandler onResponse, ErrorHandler onError)
{
    std::weak_ptr<AsyncHttpClient> self = shared_from_this();
    std::weak_ptr<AsyncHttpRequest> weakRequest = std::static_pointer_cast<AsyncHttpRequest>(request);

    auto responseHandler = [self, weakRequest, onResponse](std::shared_ptr<HttpResponse> response) {
        auto client = self.lock();
        if (client) {
            client->m_targetScheduler->schedule([client, weakRequest, onResponse, response]() {
                auto inner = weakRequest.lock();
                if (inner) {
                    std::lock_guard<RecursiveMutex> lock(inner->getMutex());
                    if (!inner->isCancelled()) {
                        onResponse(std::make_shared<AsyncHttpResponse>(inner, response, client));
                    }
                }
            });
        }
    };

    auto errorHandler = [self, weakRequest, onError](int error) {
        auto client = self.lock();
        if (client) {
            client->m_targetScheduler->schedule([weakRequest, onError, error]() {
                auto inner = weakRequest.lock();
                if (inner) {
                    std::lock_guard<RecursiveMutex> lock(inner->getMutex());
                    if (!inner->isCancelled()) {
                        onError(error);
                    }
                }
            });
        }
    };

    auto client = m_httpClient;
    m_ioScheduler->schedule([client, weakRequest, responseHandler, errorHandler]() {
        auto inner = weakRequest.lock();
        if (inner) {
            client->send(inner->getRequest(), responseHandler, errorHandler);
        }
    });
}
}
