#include "AsyncHttpResponse.hpp"
#include "AsyncHttpClient.hpp"

namespace twitch {
AsyncHttpResponse::AsyncHttpResponse(std::shared_ptr<AsyncHttpRequest> request,
    std::shared_ptr<HttpResponse> response,
    std::weak_ptr<AsyncHttpClient> client)
    : HttpResponse(response->getStatus())
    , m_request(request)
    , m_response(response)
    , m_client(std::move(client))
{
}

std::string AsyncHttpResponse::getHeader(const std::string& key) const
{
    return m_response->getHeader(key);
}

void AsyncHttpResponse::setReadTimeout(std::chrono::seconds timeout)
{
    m_response->setReadTimeout(timeout);
}

void AsyncHttpResponse::read(ContentHandler onBuffer, ErrorHandler onError)
{
    auto weakClient = m_client;
    auto request = m_request;
    auto response = m_response;

    auto errorHandler = [weakClient, request, onError](int error) {
        auto client = weakClient.lock();
        if (client) {
            client->m_targetScheduler->schedule([request, onError, error]() {
                auto inner = request.lock();
                if (inner) {
                    std::lock_guard<RecursiveMutex> lock(inner->getMutex());
                    if (!inner->isCancelled()) {
                        onError(error);
                    }
                }
            });
        }
    };

    auto bufferHandler = [weakClient, request, onBuffer](const uint8_t* buffer, size_t size, bool endOfStream) {
        auto client = weakClient.lock();
        if (!client) {
            return false;
        }

        auto sharedBuffer = std::make_shared<std::vector<uint8_t>>(buffer, buffer + size);
        client->m_targetScheduler->schedule([request, onBuffer, sharedBuffer, endOfStream]() {
            auto inner = request.lock();
            if (inner) {
                std::lock_guard<RecursiveMutex> lock(inner->getMutex());
                if (!inner->isCancelled()) {
                    onBuffer(sharedBuffer->data(), sharedBuffer->size(), endOfStream);
                }
            }
        });

        return true;
    };

    auto client = weakClient.lock();
    if (client) {
        client->m_ioScheduler->schedule([request, response, bufferHandler, errorHandler]() {
            {
                auto inner = request.lock();
                if (inner) {
                    std::lock_guard<RecursiveMutex> lock(inner->getMutex());
                    if (inner->isCancelled()) {
                        return;
                    }
                } else {
                    return;
                }
            }
            response->read(bufferHandler, errorHandler);
        });
    }
}
}
