#include "AsyncMediaPlayer.hpp"
#include "debug/trace.hpp"
#include <algorithm>

namespace twitch {
AsyncMediaPlayer::AsyncMediaPlayer(Player::Listener& listener, std::shared_ptr<twitch::NativePlatform> platform, MediaSource::Request::Listener* requestListener)
    : m_listener(listener, *this)
    , m_nativePlatform(platform)
    , m_log(platform->getLog(), "Command")
    , m_commandId(0)
{
    m_mainThreadId = std::this_thread::get_id();

    createDeferredThread();

    m_player.reset(new InternalMediaPlayer(m_listener, platform, *this));

    if (requestListener != nullptr) {
        m_player->setRequestListener(requestListener);
    }
}

AsyncMediaPlayer::~AsyncMediaPlayer()
{
    releaseDeferredThread();
}

void AsyncMediaPlayer::load(const std::string& path)
{
    addCommand(__FUNCTION__, [this, path]() {
        m_player->load(path);
    });
}

void AsyncMediaPlayer::load(const std::string& path, const std::string& mediaType)
{
    addCommand(__FUNCTION__, [this, path, mediaType]() {
        m_player->load(path, mediaType);
        updateQualitiesCache();
    });
}

void AsyncMediaPlayer::play()
{
    addCommand(__FUNCTION__, [this]() {
        m_player->play();
    });
}

void AsyncMediaPlayer::pause()
{
    addCommand(__FUNCTION__, [this]() {
        m_player->pause();
    });
}

void AsyncMediaPlayer::seekTo(MediaTime time)
{
    addCommand(__FUNCTION__, [this, time]() {
        m_player->seekTo(time);
    });
}

bool AsyncMediaPlayer::isSeekable() const
{
    safeFunction(__FUNCTION__);
    return m_cache.isSeekable;
}

MediaTime AsyncMediaPlayer::getDuration() const
{
    safeFunction(__FUNCTION__);
    return m_cache.duration;
}

MediaTime AsyncMediaPlayer::getPosition() const
{
    safeFunction(__FUNCTION__, false);
    return m_cache.position;
}

MediaTime AsyncMediaPlayer::getBufferedPosition() const
{
    safeFunction(__FUNCTION__);
    return m_player->getBufferedPosition();
}

Player::State AsyncMediaPlayer::getState() const
{
    safeFunction(__FUNCTION__);
    return m_cache.state;
}

const std::string& AsyncMediaPlayer::getName() const
{
    safeFunction(__FUNCTION__);
    return m_player->getName();
}

const std::string& AsyncMediaPlayer::getVersion() const
{
    safeFunction(__FUNCTION__);
    return m_player->getVersion();
}

bool AsyncMediaPlayer::isLooping() const
{
    safeFunction(__FUNCTION__);
    return m_player->isLooping();
}

void AsyncMediaPlayer::setLooping(bool loop)
{
    addCommand(__FUNCTION__, [this, loop]() {
        m_player->setLooping(loop);
    });
}

bool AsyncMediaPlayer::isMuted() const
{
    safeFunction(__FUNCTION__);
    return m_player->isMuted();
}

void AsyncMediaPlayer::setMuted(bool muted)
{
    addCommand(__FUNCTION__, [this, muted]() {
        m_player->setMuted(muted);
    });
}

float AsyncMediaPlayer::getVolume() const
{
    safeFunction(__FUNCTION__);
    return m_player->getVolume();
}

void AsyncMediaPlayer::setVolume(float volume)
{
    addCommand(__FUNCTION__, [this, volume]() {
        m_player->setVolume(volume);
    });
}

const Quality& AsyncMediaPlayer::getQuality() const
{
    safeFunction(__FUNCTION__);
    return m_cache.quality;
}

void AsyncMediaPlayer::setQuality(const Quality& quality)
{
    addCommand(__FUNCTION__, [this, quality]() {
        m_player->setQuality(quality);
    });
}

void AsyncMediaPlayer::setQuality(const Quality& quality, bool adaptive)
{
    addCommand(__FUNCTION__, [this, quality, adaptive]() {
        m_player->setQuality(quality, adaptive);
    });
}

const std::vector<Quality>& AsyncMediaPlayer::getQualities() const
{
    safeFunction(__FUNCTION__);
    std::lock_guard<Mutex> lock(m_cache.mutex);
    return m_cache.qualities;
}

bool AsyncMediaPlayer::getAutoSwitchQuality() const
{
    safeFunction(__FUNCTION__);
    return m_player->getAutoSwitchQuality();
}

int AsyncMediaPlayer::getAverageBitrate() const
{
    safeFunction(__FUNCTION__);
    return m_player->getAverageBitrate();
}

void AsyncMediaPlayer::setAutoSwitchQuality(bool enable)
{
    addCommand(__FUNCTION__, [this, enable]() {
        m_player->setAutoSwitchQuality(enable);
    });
}

int AsyncMediaPlayer::getBandwidthEstimate() const
{
    thisMethodShouldRunInMainThread(__FUNCTION__);
    safeFunction(__FUNCTION__);
    return 0; //[TODO] return m_player->getBandwidthEstimate();
}

float AsyncMediaPlayer::getPlaybackRate() const
{
    safeFunction(__FUNCTION__);
    return m_cache.playbackRate;
}

void AsyncMediaPlayer::setPlaybackRate(float rate)
{
    addCommand(__FUNCTION__, [this, rate]() {
        m_cache.playbackRate = rate;
        m_player->setPlaybackRate(rate);
    });
}

void AsyncMediaPlayer::setSurface(void* surface)
{
    addCommand(__FUNCTION__, [this, surface]() {
        m_player->setSurface(surface);
    });
}

std::string AsyncMediaPlayer::getPath() const
{
    safeFunction(__FUNCTION__);

    std::lock_guard<Mutex> lock(m_cache.mutex);
    return m_cache.path;
}

void AsyncMediaPlayer::setClientId(const std::string& id)
{
    addCommand(__FUNCTION__, [this, id]() {
        m_player->setClientId(id);
    });
}

void AsyncMediaPlayer::setDeviceId(const std::string& id)
{
    addCommand(__FUNCTION__, [this, id]() {
        m_player->setDeviceId(id);
    });
}

void AsyncMediaPlayer::setAuthToken(const std::string& token)
{
    addCommand(__FUNCTION__, [this, token]() {
        m_player->setAuthToken(token);
    });
}

void AsyncMediaPlayer::setPlayerType(const std::string& type)
{
    addCommand(__FUNCTION__, [this, type]() {
        m_player->setPlayerType(type);
    });
}

MediaTime AsyncMediaPlayer::getLiveLatency() const
{
    safeFunction(__FUNCTION__);
    return m_player->getLiveLatency();
}

void AsyncMediaPlayer::setAutoInitialBitrate(int bitrate)
{
    addCommand(__FUNCTION__, [this, bitrate]() {
        m_player->setAutoInitialBitrate(bitrate);
    });
}

void AsyncMediaPlayer::setAutoMaxBitrate(int bitrate)
{
    addCommand(__FUNCTION__, [this, bitrate]() {
        m_player->setAutoMaxBitrate(bitrate);
    });
}

void AsyncMediaPlayer::setAutoMaxVideoSize(int width, int height)
{
    addCommand(__FUNCTION__, [this, width, height]() {
        m_player->setAutoMaxVideoSize(width, height);
    });
}

void AsyncMediaPlayer::setLiveMaxLatency(MediaTime time)
{
    addCommand(__FUNCTION__, [this, time]() {
        m_player->setLiveMaxLatency(time);
    });
}

void AsyncMediaPlayer::setLiveLowLatencyEnabled(bool enable)
{
    addCommand(__FUNCTION__, [this, enable]() {
        m_player->setLiveLowLatencyEnabled(enable);
    });
}

void AsyncMediaPlayer::setMinBuffer(MediaTime duration)
{
    addCommand(__FUNCTION__, [this, duration]() {
        m_player->setMinBuffer(duration);
    });
}

void AsyncMediaPlayer::setMaxBuffer(MediaTime duration)
{
    addCommand(__FUNCTION__, [this, duration]() {
        m_player->setMaxBuffer(duration);
    });
}

const Statistics& AsyncMediaPlayer::getStatistics() const
{
    safeFunction(__FUNCTION__);
    return m_player->getStatistics();
}

void AsyncMediaPlayer::requestServerAd()
{
    addCommand(__FUNCTION__, [this]() {
        m_player->requestServerAd();
    });
}

void AsyncMediaPlayer::regenerateAnalyticsPlaySession()
{
    addCommand(__FUNCTION__, [this]() {
        m_player->regenerateAnalyticsPlaySession();
    });
}

void AsyncMediaPlayer::setKeepAnalyticsPlaySession(bool keepPlaySession)
{
    addCommand(__FUNCTION__, [this, keepPlaySession]() {
        m_player->setKeepAnalyticsPlaySession(keepPlaySession);
    });
}

void AsyncMediaPlayer::createDeferredThread()
{
    std::lock_guard<Mutex> lock(m_commandMutex);
    m_commandRunning = true;
    m_commandThread = std::thread(&AsyncMediaPlayer::executeDeferredCommand, this);
}

void AsyncMediaPlayer::releaseDeferredThread()
{
    {
        std::lock_guard<Mutex> lock(m_commandMutex);
        m_commandRunning = false;
        m_requestCommand.notify_one();
    }

    m_commandThread.join();
}

void AsyncMediaPlayer::addCommand(const char* functionName, std::function<void()>&& commandFunction, bool enableOutputCommand)
{
    if (!m_commandRunning) {
        return;
    }

    {
        std::lock_guard<Mutex> lock(m_commandMutex);

        AsyncCommand command;
        command.function = std::move(commandFunction);
        command.name = functionName;
        command.id = m_commandId++;
        command.enableOutputCommand = m_debuggingCommand && enableOutputCommand;

        m_commands.push(command);
    }

    m_requestCommand.notify_one();
}

void AsyncMediaPlayer::executeDeferredCommand()
{
    m_nativePlatform->setCurrentThreadName("Media Player Command Executor");

    m_commandThreadId = std::this_thread::get_id();

    while (m_commandRunning) {
        std::unique_lock<Mutex> commandLock(m_commandRequestMutex);
        m_requestCommand.wait(commandLock);

        while (true) {
            AsyncCommand currentCommand;
            {
                std::lock_guard<Mutex> lock(m_commandMutex);

                if (m_commands.empty()) {
                    break;
                }

                currentCommand = m_commands.front();
                m_commands.pop();
            }

            if (!m_disableCommandExecution) {
                if (currentCommand.enableOutputCommand) {
                    m_log.debug("(%d) [%s] is executing", currentCommand.id, currentCommand.name);

                    auto begin = std::chrono::high_resolution_clock::now();
                    currentCommand.function();
                    auto end = std::chrono::high_resolution_clock::now();

                    double deltaTime = static_cast<double>(std::chrono::duration_cast<std::chrono::microseconds>(end - begin).count());
                    m_log.debug("(%d) [%s] was executed in %.2f millisecond(s)", currentCommand.id, currentCommand.name, deltaTime / 1000.0);
                } else {
                    currentCommand.function();
                }
            }
        }
    }
}

size_t AsyncMediaPlayer::getQueueSize() const
{
    std::lock_guard<Mutex> lock(m_commandMutex);
    return m_commands.size();
}

void AsyncMediaPlayer::enableDebuggingCommand(bool enable)
{
    std::lock_guard<Mutex> lock(m_commandMutex);
    m_debuggingCommand = enable;
}

void AsyncMediaPlayer::flushAllCommands()
{
    std::lock_guard<Mutex> lock(m_commandMutex);
    std::queue<AsyncCommand> empty;
    std::swap(m_commands, empty);
}

size_t AsyncMediaPlayer::getCurrentCommandId()
{
    return m_commandId;
}

void AsyncMediaPlayer::disableCommandExecution(bool disable)
{
    flushAllCommands();

    m_disableCommandExecution = disable;
}

void AsyncMediaPlayer::thisThreadShouldRunInCommandThread(const char* function) const
{
    if (m_commandThreadId != std::this_thread::get_id()) {
        TRACE_ERROR("[ERROR] %s function should be called on command thread", function);
        assert(false);
    }
}

void AsyncMediaPlayer::thisMethodShouldRunInMainThread(const char* function) const
{
    if (m_mainThreadId != std::this_thread::get_id()) {
        TRACE_ERROR("[ERROR] %s function should be called on Main thread", function);
        assert(false);
    }
}

void AsyncMediaPlayer::triggerError(const Error& error)
{
    m_listener.onError(error);
}

void AsyncMediaPlayer::safeFunction(const char* function, bool enableOutputCommand) const
{
    if (m_debuggingCommand && enableOutputCommand) {
        m_log.debug("(%d) [%s] is executing", m_commandId.load(), function);
    }

    m_commandId++;
}

void AsyncMediaPlayer::updateCache()
{
    thisThreadShouldRunInCommandThread(__FUNCTION__);

    m_cache.isSeekable = m_player->isSeekable();

    std::string path = m_player->getPath();
    {
        std::lock_guard<Mutex> lock(m_cache.mutex);
        m_cache.path = path;
    }

    updateQualitiesCache();
}

void AsyncMediaPlayer::updateQualitiesCache()
{
    std::lock_guard<Mutex> lock(m_cache.mutex);
    m_cache.qualities = m_player->getQualities();
}

AsyncMediaPlayer::InternalMediaPlayer::InternalMediaPlayer(Player::Listener& listener, std::shared_ptr<Platform> platform, twitch::AsyncMediaPlayer& playerOwner)
    : MediaPlayer(listener, platform)
    , m_playerOwner(playerOwner)
{
}

void AsyncMediaPlayer::InternalMediaPlayer::onSourceOpened()
{
    MediaPlayer::onSourceOpened();

    m_playerOwner.addCommand(__FUNCTION__, [this]() {
        m_playerOwner.updateCache();
    });
}

void AsyncMediaPlayer::InternalMediaPlayer::onSourceSeekableChanged(bool seekable)
{
    MediaPlayer::onSourceSeekableChanged(seekable);

    m_playerOwner.addCommand(__FUNCTION__, [this]() {
        m_playerOwner.m_cache.isSeekable = m_playerOwner.m_player->isSeekable();
    });
}

void AsyncMediaPlayer::InternalMediaPlayer::onSinkFormatChanged(const MediaFormat& format)
{
    MediaPlayer::onSinkFormatChanged(format);

    m_playerOwner.addCommand(__FUNCTION__, [this]() {
        m_playerOwner.updateCache();
    });
}

AsyncMediaPlayer::Listener::Listener(Player::Listener& listener, AsyncMediaPlayer& playerOwner)
    : m_listener(listener)
    , m_playerOwner(playerOwner)
{
}

void AsyncMediaPlayer::Listener::onDurationChanged(MediaTime duration)
{
    m_playerOwner.addCommand(__FUNCTION__, [this, duration]() {
        m_playerOwner.m_cache.duration = duration;
        m_listener.onDurationChanged(duration);
    });
}

void AsyncMediaPlayer::Listener::onError(const Error& error)
{
    m_playerOwner.addCommand(__FUNCTION__, [this, error]() {
        m_playerOwner.updateQualitiesCache();
        m_listener.onError(error);
    });
}

void AsyncMediaPlayer::Listener::onMetadata(const std::string& type, const std::vector<uint8_t>& data)
{
    m_playerOwner.addCommand(__FUNCTION__, [this, type, data]() {
        m_listener.onMetadata(type, data);
    });
}

void AsyncMediaPlayer::Listener::onPositionChanged(MediaTime position)
{
    m_playerOwner.addCommand(
        __FUNCTION__, [this, position]() {
            m_playerOwner.m_cache.position = position;
            m_listener.onPositionChanged(position);
        },
        false);
}

void AsyncMediaPlayer::Listener::onQualityChanged(const Quality& quality)
{
    m_playerOwner.m_cache.quality = quality;
    m_playerOwner.addCommand(__FUNCTION__, [this]() {
        m_listener.onQualityChanged(m_playerOwner.m_cache.quality);
    });
}

void AsyncMediaPlayer::Listener::onRebuffering()
{
    m_playerOwner.addCommand(__FUNCTION__, [this]() {
        m_listener.onRebuffering();
    });
}

void AsyncMediaPlayer::Listener::onRecoverableError(const Error& error)
{
    m_playerOwner.addCommand(__FUNCTION__, [this, error]() {
        m_playerOwner.updateQualitiesCache();
        m_listener.onRecoverableError(error);
    });
}

void AsyncMediaPlayer::Listener::onSeekCompleted(MediaTime time)
{
    m_playerOwner.addCommand(__FUNCTION__, [this, time]() {
        m_listener.onSeekCompleted(time);
    });
}

void AsyncMediaPlayer::Listener::onSessionData(const std::map<std::string, std::string>& properties)
{
    m_playerOwner.addCommand(__FUNCTION__, [this, properties]() {
        m_listener.onSessionData(properties);
    });
}

void AsyncMediaPlayer::Listener::onStateChanged(State state)
{
    m_playerOwner.addCommand(__FUNCTION__, [this, state]() {
        m_playerOwner.m_cache.state = state;
        m_listener.onStateChanged(state);
    });
}

void AsyncMediaPlayer::Listener::onAnalyticsEvent(const std::string& name, const std::string& properties)
{
    m_playerOwner.addCommand(__FUNCTION__, [this, name, properties]() {
        m_listener.onAnalyticsEvent(name, properties);
    });
}
}
