#pragma once
#include "player/MediaPlayer.hpp"
#include "playercore/platform/NativePlatform.hpp"
#include <atomic>
#include <queue>

namespace twitch {
class AsyncMediaPlayer : public twitch::Player {
public:
    AsyncMediaPlayer(Player::Listener& listener, std::shared_ptr<twitch::NativePlatform> platform, MediaSource::Request::Listener* requestListener = nullptr);
    ~AsyncMediaPlayer() override;
    AsyncMediaPlayer(const AsyncMediaPlayer&) = delete;
    const AsyncMediaPlayer& operator=(const AsyncMediaPlayer&) = delete;

    void load(const std::string& path) override;
    void load(const std::string& path, const std::string& mediaType) override;
    void play() override;
    void pause() override;
    void seekTo(MediaTime time) override;
    bool isSeekable() const override;
    MediaTime getDuration() const override;
    MediaTime getPosition() const override;
    MediaTime getBufferedPosition() const override;
    Player::State getState() const override;
    const std::string& getName() const override;
    const std::string& getVersion() const override;
    bool isLooping() const override;
    void setLooping(bool loop) override;
    bool isMuted() const override;
    void setMuted(bool muted) override;
    float getVolume() const override;
    void setVolume(float volume) override;
    const Quality& getQuality() const override;
    void setQuality(const Quality& quality) override;
    void setQuality(const Quality& quality, bool adaptive) override;
    const std::vector<Quality>& getQualities() const override;
    bool getAutoSwitchQuality() const override;
    void setAutoSwitchQuality(bool enable) override;
    int getAverageBitrate() const override;
    int getBandwidthEstimate() const override;
    float getPlaybackRate() const override;
    void setPlaybackRate(float rate) override;
    void setSurface(void* surface) override;
    std::string getPath() const override;
    void setClientId(const std::string& id) override;
    void setDeviceId(const std::string& id) override;
    void setAuthToken(const std::string& token) override;
    void setPlayerType(const std::string& type) override;
    MediaTime getLiveLatency() const override;
    void setAutoInitialBitrate(int bitrate) override;
    void setAutoMaxBitrate(int bitrate) override;
    void setAutoMaxVideoSize(int width, int height) override;
    void setLiveMaxLatency(MediaTime time) override;
    void setLiveLowLatencyEnabled(bool enable) override;
    void setMinBuffer(MediaTime duration) override;
    void setMaxBuffer(MediaTime duration) override;

    const Statistics& getStatistics() const override;

    // for tv-apps
    void requestServerAd() override;
    void regenerateAnalyticsPlaySession() override;
    void setKeepAnalyticsPlaySession(bool keepPlaySession) override;

    // internal only (tests and debug: async_player_test.hpp)
    size_t getQueueSize() const;
    void enableDebuggingCommand(bool enable);
    size_t getCurrentCommandId();
    void disableCommandExecution(bool disable);
    void thisThreadShouldRunInCommandThread(const char* function) const;
    void thisMethodShouldRunInMainThread(const char* function) const;
    void triggerError(const Error& error);

private:
    void createDeferredThread();
    void releaseDeferredThread();
    void executeDeferredCommand();
    void addCommand(const char* functionName, std::function<void()>&& commandFunction, bool enableOutputCommand = true);
    void safeFunction(const char* function, bool enableOutputCommand = true) const;
    void updateCache();
    void updateQualitiesCache();
    void flushAllCommands();

private:
    class Listener : public Player::Listener {
    public:
        Listener(Player::Listener& listener, AsyncMediaPlayer& playerOwner);
        ~Listener() override = default;

        void onDurationChanged(MediaTime duration) override;
        void onError(const Error& error) override;
        void onMetadata(const std::string& type, const std::vector<uint8_t>& data) override;
        void onPositionChanged(MediaTime position) override;
        void onQualityChanged(const Quality& quality) override;
        void onRebuffering() override;
        void onRecoverableError(const Error& error) override;
        void onSeekCompleted(MediaTime time) override;
        void onSessionData(const std::map<std::string, std::string>& properties) override;
        void onStateChanged(State state) override;
        void onAnalyticsEvent(const std::string& name, const std::string& properties) override;

    private:
        Player::Listener& m_listener;
        AsyncMediaPlayer& m_playerOwner;
    };
    friend class Listener;

    struct AsyncCommand {
        std::function<void()> function;
        const char* name = nullptr;
        size_t id = 0;
        bool enableOutputCommand = false;
    };

    class InternalMediaPlayer : public twitch::MediaPlayer {
    public:
        InternalMediaPlayer(Player::Listener& listener, std::shared_ptr<Platform> platform, twitch::AsyncMediaPlayer& playerOwner);
        ~InternalMediaPlayer() override = default;
        InternalMediaPlayer(const MediaPlayer&) = delete;
        const InternalMediaPlayer& operator=(const MediaPlayer&) = delete;

    protected:
        void onSourceOpened() override;
        void onSourceSeekableChanged(bool seekable) override;
        void onSinkFormatChanged(const MediaFormat& format) override;

    private:
        twitch::AsyncMediaPlayer& m_playerOwner;
    };
    friend class InternalMediaPlayer;

    mutable Mutex m_commandMutex;
    Listener m_listener;
    std::unique_ptr<InternalMediaPlayer> m_player;
    std::shared_ptr<twitch::NativePlatform> m_nativePlatform;
    PrefixedLog m_log;

private:
    Mutex m_commandRequestMutex;
    ConditionVariable m_requestCommand;
    bool m_commandRunning = true;
    bool m_debuggingCommand = false;
    bool m_disableCommandExecution = false;

    std::queue<AsyncCommand> m_commands;

    std::thread m_commandThread;
    std::thread::id m_mainThreadId;
    std::thread::id m_commandThreadId;
    mutable std::atomic_size_t m_commandId;

    struct Cache {
        mutable Mutex mutex;
        MediaTime duration;
        MediaTime position;
        float playbackRate = 1.0f;
        bool isSeekable = false;
        std::string path;
        Quality quality;
        std::vector<Quality> qualities;
        State state = Player::State::Idle;
    };

    Cache m_cache;
};
}
