#include "BufferControl.hpp"
#include "BufferRange.hpp"
#include "media/MediaReader.hpp"
#include <cmath>

namespace twitch {
const float DefaultRate = 1.0f;

// default settings for 'standard' video playback, no speeding up, no buffer growth
BufferControl::Options BufferControl::getDefaultOptions()
{
    BufferControl::Options options;
    options.minBufferDuration = MediaTime(2.0);
    options.maxBufferDuration = MediaTime(30.0);
    options.rebufferPenalty = MediaTime::zero();
    options.liveSpeedUpRate = DefaultRate;
    return options;
}

// default settings used on the web player
BufferControl::Options BufferControl::getTwitchDefaultOptions()
{
    BufferControl::Options options;
    options.minBufferDuration = MediaTime(2.0);
    options.maxBufferDuration = MediaTime(30.0);
    options.rebufferPenalty = MediaTime(2.0);
    options.maxRebufferDuration = MediaTime(10.0);
    options.liveSpeedUpBuffer = MediaTime::max();
    options.liveSpeedUpReset = MediaTime::max();
    options.liveSpeedUpRate = 1.00f;
    return options;
}

// settings for median < 5s latency
BufferControl::Options BufferControl::getTwitchLowLatencyOptions()
{
    BufferControl::Options options = getTwitchDefaultOptions();
    options.minBufferDuration = MediaTime(1.0);
    options.rebufferPenalty = MediaTime(1.0);
    options.maxRebufferDuration = MediaTime(6.0);
    options.liveSpeedUpBuffer = MediaTime(4);
    options.liveSpeedUpReset = MediaTime(2.5);
    options.liveSpeedUpRate = 1.03f;
    return options;
}

// settings for synchronized latency demos
BufferControl::Options BufferControl::getSyncLatencyOptions()
{
    BufferControl::Options options = getTwitchLowLatencyOptions();
    options.minBufferDuration = MediaTime(1.5);
    options.rebufferPenalty = MediaTime(0.0);
    options.liveSpeedUpRate = DefaultRate;
    return options;
}

BufferControl::BufferControl(const Platform& platform)
    : m_log(platform.getLog())
    , m_frameLevelMode(platform.getCapabilities().supportsSingleFrameFragments)
    , m_options(getTwitchDefaultOptions())
    , m_optionOverrides(getDefaultOptions())
    , m_applyOverrides(false)
    , m_liveSynchronizeLatency(false)
{
    reset();
}

MediaTime BufferControl::getMinBuffer() const
{
    if (m_liveSynchronizeLatency) {
        return m_options.minBufferDuration;
    } else {
        MediaTime rebuffer = m_options.minBufferDuration + (m_options.rebufferPenalty * m_rebufferCount);
        return m_options.maxRebufferDuration > MediaTime::zero()
            ? std::min(m_options.maxRebufferDuration, rebuffer)
            : rebuffer;
    }
}

MediaTime BufferControl::getMaxBuffer() const
{
    return m_options.maxBufferDuration;
}

MediaTime BufferControl::getFillTime() const
{
    if (m_bufferState == BufferState::Draining) {
        return m_bufferingTime.duration;
    } else {
        return MediaTime::now() - m_bufferingTime.start;
    }
}

TimeRange BufferControl::getBufferedRange(MediaTime position) const
{
    for (const auto& range : m_ranges) {
        if (range.end.valid() && range.end != MediaTime::zero()) {
            if (range.contains(position)) {
                return TimeRange(range.start, range.duration());
            }
        }
    }
    return TimeRange();
}

TimeRange BufferControl::getBufferTrimRange(MediaTime position)
{
    // trim buffer data before the playable range, only 1 buffered range is supported
    // a multiplier is used to set the trimming threshold to prevent too many removes
    TimeRange range;
    const float BackBufferVariance = 1.2f;
    MediaTime maxBackBuffer = getMaxBuffer() * BackBufferVariance;
    if (position > maxBackBuffer) {
        TimeRange buffered = getBufferedRange(position);
        if (buffered.duration > MediaTime::zero() && (position - buffered.start) > maxBackBuffer) {
            range = TimeRange(MediaTime::zero(), position - maxBackBuffer);
            // assume the buffer start to be ahead of what was removed and not exact (for web)
            MediaTime start = position - getMaxBuffer();
            if (buffered.start > MediaTime::zero() && start > buffered.start) {
                setBufferStart(start);
            }
        }
    }
    return range;
}

MediaTime BufferControl::getBufferEnd() const
{
    if (m_ranges.empty()) {
        return MediaTime::zero();
    }
    return m_ranges.back().end;
}

MediaTime BufferControl::getSyncTimeBetween(MediaTime start, MediaTime end) const
{
    MediaTime result = MediaTime::invalid();
    for (const auto& sync : m_syncTimes) {
        if (sync > start) {
            result = sync;
        }
        if (sync > end) {
            break;
        }
    }
    return result;
}

MediaTime BufferControl::getBufferReadTimeout() const
{
    const MediaTime DefaultTimeout(10.0);
    return m_liveSynchronizeLatency ? getMinBuffer() : DefaultTimeout;
}

TimeRange BufferControl::getPlayableRange(MediaTime position) const
{
    MediaTime start = MediaTime::zero();
    MediaTime duration = MediaTime::zero();
    for (const auto& range : m_ranges) {
        // ensure the position is inside the buffered range
        start = range.start.valid() ? std::max(position, range.start) : MediaTime::zero();
        if (range.end >= start) {
            duration = range.end - start;
            break;
        }
    }

    // special case transition from one buffer range to another
    if (duration < getMinBuffer() && m_ranges.size() > 1) {
        const auto& range = m_ranges.back();
        if (range.start.valid()) {
            return TimeRange(range.start, range.end - range.start);
        }
    }

    return TimeRange(start, duration);
}

bool BufferControl::isBufferingTimedOut(MediaTime latency) const
{
    if (m_liveSynchronizeLatency) {
        const float LatencyFactor = 1.1f;
        return getFillTime() > (getMinBuffer() * LatencyFactor);
    } else if (m_liveMaxLatency > MediaTime::zero() && latency > MediaTime::zero()) {
        return getFillTime() + latency > m_liveMaxLatency;
    } else {
        return false;
    }
}

void BufferControl::setBufferStart(MediaTime time)
{
    if (!m_ranges.empty()) {
        auto& range = m_ranges.back();
        if (range.contains(time)) {
            range.start = time;
            if (range.start > range.end) {
                m_log->warn("buffer start %.2f > end %.2f", time.seconds(), range.end.seconds());
                range.end = time;
            }
        }
    }
}

void BufferControl::setBufferEnd(bool discontinuity, MediaTime time)
{
    if (m_ranges.empty()) {
        m_ranges.emplace_back(time, time);
        return;
    } else if (discontinuity) {
        auto& range = m_ranges.back();
        if ((range.end - time).absolute() > MediaTime(3.0)) {
            m_ranges.emplace_back(time, time);
            m_log->info("buffer range discontinuity start from %.3f us end %.3f", time.seconds(), range.end.seconds());
            logRanges();
            return;
        }
    }
    auto& range = m_ranges.back();
    if (time >= range.end) {
        range.end = time;
    }
}

void BufferControl::setSyncTime(MediaTime time)
{
    m_syncTimes.push_back(time);
    // limit the sync point history
    if (m_syncTimes.back() - m_syncTimes.front() > getMaxBuffer()) {
        m_syncTimes.erase(m_syncTimes.begin());
    }
}

void BufferControl::setMinBuffer(MediaTime duration)
{
    m_optionOverrides.minBufferDuration = duration;
    m_applyOverrides = true;
    m_options = m_optionOverrides;
}

void BufferControl::setMaxBuffer(MediaTime duration)
{
    m_optionOverrides.maxBufferDuration = duration;
    m_applyOverrides = true;
    m_options = m_optionOverrides;
}

void BufferControl::setLiveSpeedUpRate(float rate)
{
    m_options.liveSpeedUpRate = rate;
}

void BufferControl::setState(BufferState state)
{
    if (state != m_bufferState) {
        m_log->info("BufferState changed %s", bufferStateToString(state));
        m_bufferState = state;
        switch (m_bufferState) {
        case BufferState::Uninitialized:
            break;
        case BufferState::Filling:
            m_bufferingTime.start = MediaTime::now();
            break;
        case BufferState::Refilling:
            m_bufferingTime.start = MediaTime::now();
            m_rebufferCount++;
            break;
        case BufferState::Draining:
            m_bufferingTime.duration = MediaTime::now() - m_bufferingTime.start;
            break;
        }
    }
}

void BufferControl::reset()
{
    m_bufferState = BufferState::Uninitialized;
    m_bufferingTime = TimeRange();
    m_ranges.clear();
    m_syncTimes.clear();
    m_rebufferCount = 0;
    m_speedUpRate = DefaultRate;
    m_position = MediaTime::invalid();
}

float BufferControl::getSpeedUpRate(MediaTime position)
{
    if (m_options.liveSpeedUpRate != DefaultRate) {
        TimeRange range = getPlayableRange(position);
        // Shift speedup window by rebuffer penalty
        MediaTime penalty = getMinBuffer() - m_options.minBufferDuration;
        MediaTime low = m_options.liveSpeedUpReset + penalty;
        MediaTime high = m_options.liveSpeedUpBuffer + penalty;
        if (range.duration > high) {
            m_speedUpRate = m_options.liveSpeedUpRate;
        } else if (range.duration < low) {
            m_speedUpRate = DefaultRate;
        }
    } else {
        m_speedUpRate = DefaultRate;
    }

    return m_speedUpRate;
}

void BufferControl::seekTo(MediaTime time)
{
    // if the time is outside the buffered range reset the buffered range
    if (!getBufferedRange(time).contains(time)) {
        m_ranges.clear(); // eventually we'll support multiple ranges
    }

    m_speedUpRate = DefaultRate;
}

void BufferControl::updateBufferEnd(int track, const MediaSample& sample)
{
    using namespace twitch::media;
    // update the end time of the buffer range
    if (m_frameLevelMode) {
        if (track != media::MediaReader::MetaTrackId
            && track != media::MediaReader::TextTrackId
            && sample.duration > MediaTime::zero()) {
            setBufferEnd(sample.isDiscontinuity, sample.presentationTime);
        }
    } else if (sample.isSyncSample) {
        setBufferEnd(sample.isDiscontinuity, sample.presentationTime);
    }

    if (track == media::MediaReader::VideoTrackId && sample.isSyncSample) {
        setSyncTime(sample.decodeTime);
    }
}

void BufferControl::updatePosition(MediaTime position)
{
    if (position.valid() && m_ranges.size() > 1) {
        const auto& current = m_ranges[0];
        const auto& next = m_ranges[1];
        if (next.contains(position)) {
            // if position outside current buffer but is in next buffer
            if (!current.contains(position)
                // if position jumps backwards move to next buffered range
                || (m_position.valid() && position < m_position)
                // position moved past end of buffer
                || (current.end.valid() && position > current.end)) {
                m_ranges.erase(m_ranges.begin());
            }
        }
    }

    m_position = position;
}

void BufferControl::update(const Settings& settings, bool isLowLatency)
{
    std::string group = isLowLatency ? "lowlatency" : "buffercontrol";
    // reset the default values (if no settings)
    if (m_applyOverrides) {
        m_options = m_optionOverrides;
        return; // settings ignored if buffer settings overriden through the API
    } else if (isLowLatency) {
        if (isSynchronizedLatencyMode()) {
            m_options = getSyncLatencyOptions();
        } else {
            m_options = getTwitchLowLatencyOptions();
        }
    } else {
        m_options = getTwitchDefaultOptions();
    }

    auto getTime = [&group, &settings](const std::string& key, const MediaTime& defaultValue) -> MediaTime {
        return MediaTime(settings.get(group, key, defaultValue.seconds()));
    };
    m_options.minBufferDuration = getTime("minBufferDuration", m_options.minBufferDuration);
    m_options.maxBufferDuration = getTime("maxBufferDuration", m_options.maxBufferDuration);
    m_options.rebufferPenalty = getTime("rebufferPenalty", m_options.rebufferPenalty);
    m_options.maxRebufferDuration = getTime("maxRebufferDuration", m_options.maxRebufferDuration);
    m_options.liveSpeedUpBuffer = getTime("liveSpeedUpBuffer", m_options.liveSpeedUpBuffer);
    m_options.liveSpeedUpReset = getTime("liveSpeedUpReset", m_options.liveSpeedUpReset);
    m_options.liveSpeedUpRate = settings.get(group, "liveSpeedUpRate", m_options.liveSpeedUpRate);
}

void BufferControl::logRanges() const
{
    std::string list;
    for (const auto& range : m_ranges) {
        if (!list.empty()) {
            list += ", ";
        }
        list += std::to_string(range.start.seconds()) + "-" + std::to_string(range.end.seconds());
    }
    m_log->info("buffer ranges %s", list.c_str());
}
}
