#pragma once

#include "BufferRange.hpp"
#include "BufferState.hpp"
#include "Settings.hpp"
#include "playercore/MediaSample.hpp"
#include "playercore/MediaTime.hpp"
#include "playercore/TimeRange.hpp"
#include "playercore/platform/Platform.hpp"

namespace twitch {
/** Decides the current minimum and maximum buffer sizes for the player */
class BufferControl {
public:
    explicit BufferControl(const Platform& platform);
    virtual ~BufferControl() = default;
    BufferControl(const BufferControl&) = delete;
    const BufferControl& operator=(const BufferControl&) = delete;

    MediaTime getMinBuffer() const;
    MediaTime getMaxBuffer() const;
    MediaTime getBufferReadTimeout() const;
    MediaTime getFillTime() const;
    TimeRange getBufferedRange(MediaTime position) const;
    TimeRange getBufferTrimRange(MediaTime position);
    MediaTime getBufferEnd() const;
    TimeRange getPlayableRange(MediaTime position) const;
    int getRebufferCount() const { return m_rebufferCount; }
    MediaTime getSyncTimeBetween(MediaTime start, MediaTime end) const;
    float getSpeedUpRate(MediaTime position);
    BufferState getState() const { return m_bufferState; }
    bool isBufferingTimedOut(MediaTime latency) const;
    bool isSynchronizedLatencyMode() const { return m_liveSynchronizeLatency; }
    void setSynchronizedLatencyMode(bool enable) { m_liveSynchronizeLatency = enable; }
    bool isFrameLevelMode() { return m_frameLevelMode; }
    void setLiveMaxLatency(MediaTime time) { m_liveMaxLatency = time; }
    void setMinBuffer(MediaTime duration);
    void setMaxBuffer(MediaTime duration);
    void setLiveSpeedUpRate(float rate);
    void setState(BufferState state);
    void reset();
    void seekTo(MediaTime time);
    void updatePosition(MediaTime position);
    void updateBufferEnd(int track, const MediaSample& sample);
    void update(const Settings& settings, bool isLowLatency);

private:
    void setBufferStart(MediaTime time);
    void setBufferEnd(bool discontinuity, MediaTime time);
    void setSyncTime(MediaTime time);
    void logRanges() const;

    struct Options {
        MediaTime minBufferDuration; /** Minimum playback buffer size */
        MediaTime maxBufferDuration; /** Maximum playback buffer size */
        MediaTime rebufferPenalty; /** Duration to add to min buffer on rebuffer */
        MediaTime maxRebufferDuration; /** Maximum playback buffer size on a rebuffer */
        MediaTime liveSpeedUpBuffer; /** Buffer size to trigger speed up */
        MediaTime liveSpeedUpReset; /** Buffer size to remove speed up */
        float liveSpeedUpRate = 1.00f; /** Live speed up playback rate multiplier */
    private:
        Options() {};
        friend class BufferControl;
    };

    static Options getDefaultOptions();
    static Options getTwitchDefaultOptions();
    static Options getTwitchLowLatencyOptions();
    static Options getSyncLatencyOptions();

    std::shared_ptr<Log> m_log;
    bool m_frameLevelMode;
    Options m_options;
    Options m_optionOverrides;
    bool m_applyOverrides;
    std::vector<BufferRange> m_ranges;
    std::vector<MediaTime> m_syncTimes;
    MediaTime m_liveMaxLatency;
    MediaTime m_position;
    BufferState m_bufferState;
    int m_rebufferCount;
    float m_speedUpRate;
    bool m_liveSynchronizeLatency;
    TimeRange m_bufferingTime;
};
}
