#pragma once

#include "debug/ThreadGuard.hpp"
#include "playercore/Player.hpp"
#include <functional>

namespace twitch {
class CompositeListener : public Player::Listener {
public:
    template <typename... Args>
    explicit CompositeListener(Args&&... args)
    {
        m_listeners = { args... };
    }

    template <typename Method, typename... Args>
    void invoke(Method m, Args&&... args)
    {
        m_threadGuard.validate();

        for (const auto& ref : m_listeners) {
            (ref.get().*m)(std::forward<Args>(args)...);
        }
    }

    void onDurationChanged(MediaTime duration) override
    {
        invoke(&Player::Listener::onDurationChanged, duration);
    }

    void onError(const Error& error) override
    {
        invoke(&Player::Listener::onError, error);
    }

    void onMetadata(const std::string& type, const std::vector<uint8_t>& data) override
    {
        invoke(&Player::Listener::onMetadata, type, data);
    }

    void onPositionChanged(MediaTime position) override
    {
        // not on player thread for performance reasons
        for (const auto& ref : m_listeners) {
            ref.get().onPositionChanged(position);
        }
    }

    void onQualityChanged(const Quality& quality) override
    {
        invoke(&Player::Listener::onQualityChanged, quality);
    }

    void onRebuffering() override
    {
        invoke(&Player::Listener::onRebuffering);
    }

    void onRecoverableError(const Error& error) override
    {
        invoke(&Player::Listener::onRecoverableError, error);
    }

    void onSeekCompleted(MediaTime time) override
    {
        invoke(&Player::Listener::onSeekCompleted, time);
    }

    void onSessionData(const std::map<std::string, std::string>& properties) override
    {
        invoke(&Player::Listener::onSessionData, properties);
    }

    void onStateChanged(Player::State state) override
    {
        invoke(&Player::Listener::onStateChanged, state);
    }

    void onAnalyticsEvent(const std::string& name, const std::string& properties) override
    {
        invoke(&Player::Listener::onAnalyticsEvent, name, properties);
    }

private:
    std::vector<std::reference_wrapper<Player::Listener>> m_listeners;
    SingleThreadGuard m_threadGuard;
};
}
