#include "MediaPlayer.hpp"
#include "debug/TraceLog.hpp"
#include "media/CodecString.hpp"
#include "media/fourcc.hpp"
#include "twitch/ChannelSource.hpp"
#include "twitch/ClipSource.hpp"
#include <algorithm>

namespace twitch {
MediaPlayer::MediaPlayer(Player::Listener& listener, std::shared_ptr<Platform> platform)
    : ScopedScheduler(platform->createScheduler("Player Scheduler"))
    , m_analytics(new analytics::AnalyticsTracker(*this, listener, platform, share()))
    , m_listener(*m_analytics, listener)
    , m_requestListener(nullptr)
    , m_platform(platform)
    , m_tokenHandler(std::make_shared<TokenHandler>(platform))
    , m_volume(1.0)
    , m_muted(false)
    , m_state(Player::State::Idle)
    , m_buffer(*platform)
    , m_log(platform->getLog(), "Player ")
    , m_looping(false)
    , m_paused(true)
    , m_autoSwitchQuality(true)
    , m_resumable(false)
    , m_remote(false)
    , m_qualitySelector(platform)
    , m_liveLowLatencyEnabled(true)
    , m_hidden(false)
    , m_changedVisibleQuality(false)
    , m_viewportSize(0, 0)
    , m_surface(nullptr)
{
    // TODO eventually remove global log needed for the TRACE_ macro calls
    TraceLog::get().setLog(platform->getLog());
    m_sink = createSink();
    m_log.debug("created");
}

MediaPlayer::~MediaPlayer()
{
    m_log.debug("destructor");
    scheduleAndWait([this]() {
        m_cancelRead.cancel();
        m_source.clear();
        m_sink.reset();
        m_analytics.reset();
    });
    cancel();
}

void MediaPlayer::load(const std::string& path)
{
    load(path, "");
}

void MediaPlayer::load(const std::string& path, const std::string& mediaType)
{
    m_log.debug("load %s", path.c_str());
    scheduleAndWait([this, path, mediaType] {
        m_path = path;
        m_mediaType = mediaType;
        handleClose(!m_remote);
        handleOpen(path, MediaType(mediaType));
        if (m_analytics) {
            m_analytics->setEnabled(!m_remote);
            m_analytics->onPlayerLoad(path);
        }
    });
}

void MediaPlayer::play()
{
    m_log.debug("play");
    scheduleAndWait([this]() {
        if (m_state == Player::State::Ended) {
            if (m_source.isLive()) {
                return;
            } else if (!m_playhead.isSeeking()) {
                handleSeek(MediaTime::zero(), true);
            }
        }
        m_paused = false;
        scheduleRead();
        // if there is already enough buffered play immediately
        checkPlayable();
    });
}

void MediaPlayer::pause()
{
    m_log.debug("pause");
    scheduleAndWait([this]() {
        m_paused = true;
        bool ended = m_state == Player::State::Ended;
        handleClose(false, ended ? Player::State::Ended : Player::State::Idle);
    });
}

bool MediaPlayer::isSeekable() const
{
    bool seekable = false;
    scheduleAndWait([this, &seekable]() {
        seekable = m_source.isSeekable();
    });
    return seekable;
}

void MediaPlayer::seekTo(MediaTime time)
{
    m_log.debug("seekTo %lld us", time.microseconds().count());
    scheduleAndWait([this, time]() {
        auto range = m_buffer.getPlayableRange(m_playhead.getPosition());
        // Currently only web keeps a back buffer, this will change for the native platforms in the future
        if (m_platform->getName() == "web") {
            MediaTime start = std::max(m_playhead.getPosition() - m_buffer.getMaxBuffer(), MediaTime::zero());
            // cap back buffer start to the max buffer behind the playhead, as data before that will be evicted
            range = m_buffer.getBufferedRange(range.start);
            if (start > range.start) {
                range = TimeRange(start, range.end() - start);
            }
        }
        // seek ahead within the buffer if possible
        if (m_source.isPassthrough()
            || (range.contains(time) && (m_source.isEnded() || range.end() - m_buffer.getMinBuffer() > time))) {
            m_sink->pause();
            m_sink->seekTo(time);
            m_playhead.seekTo(time);
            schedule([this]() {
                checkPlayable();
            });
        } else {
            m_analytics->onPlayerSeek(m_playhead.getPosition(), time);
            handleSeek(time, true);
        }
    });
}

MediaTime MediaPlayer::getDuration() const
{
    return m_source.getDuration();
}

MediaTime MediaPlayer::getPosition() const
{
    return m_playhead.getPosition();
}

MediaTime MediaPlayer::getBufferedPosition() const
{
    return m_buffer.getPlayableRange(m_playhead.getPosition()).end();
}

const std::string& MediaPlayer::getName() const
{
    static const std::string Name("mediaplayer");
    return Name;
}

const std::string& MediaPlayer::getVersion() const
{
    static const std::string LibraryVersion(PLAYERCORE_VERSION);
    return LibraryVersion;
}

void MediaPlayer::setSurface(void* surface)
{
    scheduleAndWait([=]() {
        m_surface = surface;
        m_sink->setSurface(surface);
    });
}

void MediaPlayer::setPlaybackRate(float rate)
{
    schedule([=]() {
        m_log.debug("setPlaybackRate to %f", rate);
        m_playhead.setPlaybackRate(rate);
        m_qualitySelector.setPlaybackRate(rate);
        m_sink->setPlaybackRate(rate);
    });
}

std::string MediaPlayer::getPath() const
{
    std::string path;
    scheduleAndWait([this, &path]() {
        path = m_source.getPath();
    });
    return path;
}

void MediaPlayer::setClientId(const std::string& id)
{
    schedule([=]() {
        m_tokenHandler->setClientId(id);
    });
}

void MediaPlayer::setDeviceId(const std::string& id)
{
    schedule([=]() {
        m_tokenHandler->setUniqueId(id);
        m_analytics->setDeviceId(id);
    });
}

void MediaPlayer::setAuthToken(const std::string& token)
{
    schedule([=]() {
        m_tokenHandler->setAuthToken(token);
    });
}

void MediaPlayer::setAutoInitialBitrate(int bitrate)
{
    schedule([=]() {
        m_qualitySelector.setInitialBitrate(bitrate);
    });
}

void MediaPlayer::setAutoMaxBitrate(int bitrate)
{
    schedule([=]() {
        m_qualitySelector.setMaxBitrate(bitrate);
    });
}

void MediaPlayer::setAutoMaxVideoSize(int width, int height)
{
    schedule([=]() {
        m_qualitySelector.setMaxVideoSize(width, height);
    });
}

void MediaPlayer::setAutoViewportSize(int width, int height)
{
    schedule([=]() {
        media::Resolution size(width, height);

        if (!m_sessionData.isHoldbackGroup()) {
            m_qualitySelector.setViewportSize(width, height);
            if (m_viewportSize.area() > 0 && size > m_viewportSize) {
                replaceBuffer(false);
            }
        }

        m_viewportSize = size;
    });
}

void MediaPlayer::setAutoViewportScale(float scale)
{
    schedule([=]() {
        m_qualitySelector.setViewportScale(scale);
    });
}

void MediaPlayer::setLiveMaxLatency(MediaTime time)
{
    schedule([=]() {
        m_buffer.setLiveMaxLatency(time);
    });
}

void MediaPlayer::setLiveLowLatencyEnabled(bool enable)
{
    schedule([=]() {
        if (m_liveLowLatencyEnabled != enable) {
            m_liveLowLatencyEnabled = enable;
            updateBufferOptions(enable && m_sessionData.isLowLatency());
            m_source.setLowLatencyEnabled(enable);
            m_source.setQuality(m_qualities.getSelected(), m_autoSwitchQuality);
            handleSeekToDefault();
        }
    });
}

void MediaPlayer::setMinBuffer(MediaTime duration)
{
    schedule([=]() {
        m_buffer.setMinBuffer(duration);
    });
}

void MediaPlayer::setMaxBuffer(MediaTime duration)
{
    schedule([=]() {
        m_buffer.setMaxBuffer(duration);
    });
}

void MediaPlayer::startRemotePlayback()
{
    scheduleAndWait([this] {
        m_remote = true;

        bool wasVOD = !m_source.isLive();
        MediaTime position = m_playhead.getPosition();

        handleClose(true);

        if (wasVOD) {
            m_playhead.seekTo(position);
        }
        m_analytics->setEnabled(false);

        auto source = createSource(m_path, m_mediaType, true);
        m_source.clear(); // only supporting one source at time currently
        m_source.add(m_path, std::move(source));
        m_source.open();
    });
}

void MediaPlayer::endRemotePlayback()
{
    scheduleAndWait([this] {
        // Do nothing if not already remote playback
        if (!m_remote) {
            return;
        }

        m_remote = false;

        bool wasVOD = !m_source.isLive();
        MediaTime position = m_playhead.getPosition();

        handleClose(true);

        if (wasVOD) {
            m_playhead.seekTo(position);
        }
        m_analytics->setEnabled(true);

        auto source = createSource(m_path, m_mediaType);
        m_source.clear(); // only supporting one source at time currently
        m_source.add(m_path, std::move(source));
        m_source.open();
    });
}

std::shared_ptr<Platform> MediaPlayer::getPlatform() const
{
    return m_platform;
}

void MediaPlayer::setAnalyticsSendEvents(bool enable)
{
    schedule([=]() {
        m_analytics->setSendEvents(enable);
    });
}

void MediaPlayer::setAnalyticsEndpoint(const std::string& endpoint)
{
    schedule([=]() {
        m_analytics->setAnalyticsEndpoint(endpoint);
    });
}

void MediaPlayer::setMuted(bool muted)
{
    schedule([=]() {
        m_log.debug("setMuted to %s", muted ? "true" : "false");
        m_muted = muted;
        m_sink->setVolume(m_muted ? 0.0f : m_volume);
    });
}

void MediaPlayer::requestServerAd()
{
    if (!m_sessionData.getAdTriggerUrl().empty() && TwitchLink(m_path).isChannel()) {
        auto* source = static_cast<ChannelSource*>(m_source.getCurrentSource());
        if (source) {
            source->requestServerAd(m_sessionData.getAdTriggerUrl());
        }
    }
}

void MediaPlayer::regenerateAnalyticsPlaySession()
{
    // Generate a new PlaySession (assuming one is already active)
    scheduleAndWait([this]() {
        m_analytics->onRegeneratePlaySession();
    });
}

void MediaPlayer::setKeepAnalyticsPlaySession(bool keepPlaySession)
{
    scheduleAndWait([this, keepPlaySession]() {
        m_analytics->setKeepPlaySession(keepPlaySession);
    });
}

void MediaPlayer::setLiveSpeedUpRate(float rate)
{
    schedule([=]() {
        m_buffer.setLiveSpeedUpRate(rate);
    });
}

void MediaPlayer::setPlayerType(const std::string& type)
{
    schedule([=]() {
        m_tokenHandler->setPlayerType(type);
    });
}

void MediaPlayer::setSettings(const json11::Json& json)
{
    m_settings.load(json);

    auto level = m_settings.get("logging", "level", std::string());
    if (!level.empty()) {
        m_log.setLevel(Log::levelFromString(level));
    }
}

void MediaPlayer::setVisible(bool visible)
{
    m_cancelHidden.cancel();

    if (m_source.isLive() && !m_sessionData.isHoldbackGroup()) {
        if (visible) {
            setHidden(false);
        } else {
            m_cancelHidden = schedule([this]() { setHidden(true); }, std::chrono::seconds(60));
        }
    }
}

void MediaPlayer::setVolume(float volume)
{
    schedule([=]() {
        m_log.debug("setVolume to %f", volume);
        m_volume = volume;
        if (!m_muted) {
            m_sink->setVolume(m_volume);
        }
    });
}

const Quality& MediaPlayer::getQuality() const
{
    return m_qualities.getSelected().name.empty() ? m_qualities.getCurrent() : m_qualities.getSelected();
}

void MediaPlayer::setQuality(const Quality& quality)
{
    setQuality(quality, false);
}

void MediaPlayer::setQuality(const Quality& quality, bool adaptive)
{
    scheduleAndWait([=]() {
        m_autoSwitchQuality = false;
        if (m_qualities.getSelected() != quality && !m_qualities.isEmpty()) {

            if (adaptive) {
                Quality target = m_qualities.match(quality);
                m_qualities.setSelected(target);
                m_source.setQuality(m_qualities.getSelected(), true);
            } else {
                updateSourceQuality(quality);
                // only load content if a vod or not paused
                handleSeekToDefault();
            }
        }
    });
}

const std::vector<Quality>& MediaPlayer::getQualities() const
{
    return m_qualities.getQualities();
}

bool MediaPlayer::getAutoSwitchQuality() const
{
    return m_autoSwitchQuality;
}

void MediaPlayer::setAutoSwitchQuality(bool enable)
{
    schedule([=]() {
        m_log.debug("setAutoSwitchQuality to %s", std::to_string(enable).c_str());
        if (m_autoSwitchQuality != enable) {
            m_autoSwitchQuality = enable;
            if (m_sessionData.isLowLatency() || !m_qualities.getSelected().autoSelect || m_source.isPassthrough()) {
                // reset buffer options
                // until low latency ABR works on all platforms update the quality when turning auto on/off
                bool supportsLowLatencyABR = m_platform->getCapabilities().supportsLowLatencyABR;
                updateBufferOptions(!m_autoSwitchQuality || supportsLowLatencyABR);
                m_source.setQuality(m_qualities.getSelected(), m_autoSwitchQuality);
                handleSeekToDefault();
            }
        }
    });
}

int MediaPlayer::getAverageBitrate() const
{
    int bitrate;
    scheduleAndWait([this, &bitrate]() {
        bitrate = m_qualitySelector.getAverageBitrate();
    });
    return bitrate;
}

int MediaPlayer::getBandwidthEstimate() const
{
    int estimate;
    scheduleAndWait([this, &estimate]() {
        estimate = m_qualitySelector.getBandwidthEstimate();
    });
    return estimate;
}

MediaTime MediaPlayer::getLiveLatency() const
{
    return m_liveStatistics.getBroadcasterLatency();
}

float MediaPlayer::getPlaybackRate() const
{
    return m_playhead.getPlaybackRate();
}

std::unique_ptr<MediaSource> MediaPlayer::createSource(const std::string& path, const MediaType& mediaType, bool remote)
{
    std::unique_ptr<MediaSource> source;
    TwitchLink link(path);

    if (link.isChannel() || TwitchLink::isUsherUrl(path)) { // needed to handle drm links and mobile apps
        hls::HlsSource::Options options;
        if (m_buffer.isSynchronizedLatencyMode()) {
            options.liveWindowSegments = 0;
        } else if (m_platform->getName() == "web" && m_buffer.isFrameLevelMode()) {
            options.liveWindowSegments = 2;
        }
        source.reset(new ChannelSource(*this, m_platform, share(), m_tokenHandler, path, options, remote));
    } else if (link.isClip()) {
        source.reset(new ClipSource(*this, m_platform, share(), link));
    } else {
        // loading a MP4 url or file, or non twitch HLS stream
        MediaType type = mediaType.name.empty() ? MediaType::matchFromPath(path) : mediaType;
        source = m_platform->createSource(path, type, *this, share());
        // handle non-twitch HLS stream
        if (!source && type.matches(MediaType::Application_MPEG_URL)) {
            hls::HlsSource::Options options;
            auto httpClient = m_platform->createAsyncHttpClient(share());
            source.reset(new hls::HlsSource(*this, m_platform, share(), httpClient, path, options));
        }
    }

    if (source) {
        source->setReadTimeout(m_buffer.getBufferReadTimeout());
    }

    return source;
}

std::unique_ptr<MediaSink> MediaPlayer::createSink()
{
    std::unique_ptr<MediaSink> sink = m_platform->createSink(*this, ScopedScheduler::share());

    if (sink) {
        // set initial playback settings
        sink->setPlaybackRate(m_playhead.getPlaybackRate());
        sink->setVolume(m_muted ? 0.0f : m_volume);
        sink->setSurface(m_surface);
        if (m_playhead.isSeeking()) {
            sink->seekTo(m_playhead.getPosition());
        }
    }

    return sink;
}

void MediaPlayer::updateState(Player::State state)
{
    if (m_state != state) {
        m_log.info("state changing %s to %s", stateToString(m_state), stateToString(state));
        m_state = state;
        m_listener.onStateChanged(state);
    }
}

void MediaPlayer::scheduleRead(MediaTime delay)
{
    m_cancelRead.cancel();
    m_cancelRead = schedule([this]() { handleRead(); }, delay.microseconds());
}

void MediaPlayer::onRequestSent(const MediaSource::Request& request)
{
    m_qualitySelector.onRequestSent(request);
    if (m_requestListener) {
        m_requestListener->onRequestSent(request);
    }
}

void MediaPlayer::onResponseReceived(const MediaSource::Request& request)
{
    m_qualitySelector.onResponseReceived(request);
    if (m_requestListener) {
        m_requestListener->onResponseReceived(request);
    }
}

void MediaPlayer::onResponseBytes(const MediaSource::Request& request, size_t bytes)
{
    m_qualitySelector.onResponseBytes(request, bytes);
    if (m_requestListener) {
        m_requestListener->onResponseBytes(request, bytes);
    }
}

void MediaPlayer::onResponseEnd(const MediaSource::Request& request)
{
    m_qualitySelector.onResponseEnd(request);
    if (m_requestListener) {
        m_requestListener->onResponseEnd(request);
    }
}

void MediaPlayer::onRequestError(const MediaSource::Request& request, int error)
{
    m_qualitySelector.onRequestError(request, error);
    if (m_requestListener) {
        m_requestListener->onRequestError(request, error);
    }
}

void MediaPlayer::onSourceDurationChanged(MediaTime duration)
{
    m_source.onDurationChanged(duration);
    m_qualitySelector.setStreamType(m_source.isLive() ? abr::QualitySelector::StreamType::Live
                                                      : abr::QualitySelector::StreamType::VOD);
    m_listener.onDurationChanged(duration);

    // if not paused or not live (but auto) prebuffer content
    // if not auto this is disabled so the player caller can select an initial quality
    if (!m_paused || (!m_source.isLive() && (m_autoSwitchQuality || m_qualities.getQualities().size() <= 1))) {
        scheduleRead();
    }
}

void MediaPlayer::onSourceSeekableChanged(bool seekable)
{
    m_source.onSeekableChanged(seekable);
}

void MediaPlayer::onSourceEndOfStream()
{
    m_log.info("Source end of stream");
    m_source.onEndOfStream(m_buffer.getBufferEnd());

    if (m_source.isEnded()) {
        m_sink->endOfStream();
        if (m_source.isPassthrough()) {
            if (m_looping && m_source.isSeekable()) {
                handleSeek(MediaTime::zero(), true);
                if (!m_paused) {
                    m_sink->play();
                }
            } else {
                updateState(Player::State::Ended);
            }
        } else {
            // play out any remaining data in the buffer
            // if the data was already played out transition to the ended state
            if (!checkPlayable() && m_state != Player::State::Ready && m_state != Player::State::Playing) {
                updateState(Player::State::Ended);
            }
        }
    }
}

void MediaPlayer::onSourceError(const Error& error)
{
    // schedule handling the error since it will close the source (possibly destroying it)
    schedule([=]() {
        handleError(error);
    });
}

void MediaPlayer::onSourceRecoverableError(const Error& error)
{
    m_log.warn("recoverable error %s:%d (%s code %d - %s)",
        errorSourceString(error.source), error.result.value,
        mediaResultString(error.result), error.result.code, error.message.c_str());

    m_listener.onRecoverableError(error);
}

void MediaPlayer::onSourceFlush()
{
    m_source.onFlush();
    m_sink->flush();

    bool playable = false;

    if (m_state != Player::State::Playing) {
        if (m_buffer.getState() != BufferState::Draining) {
            playable = checkPlayable();
        }
        // re-check the state which could have changed from checkPlayable
        if (m_paused && (m_state == Player::State::Buffering || m_state == Player::State::Ready)) {
            // if buffer isn't playable yet read more
            if (!playable && !m_source.isLive()) {
                handleRead();
                return;
            }
        } else if (!m_paused && m_state != Player::State::Playing) {
            // if buffer isn't playable enter the buffering state
            updateState(Player::State::Buffering);
        }
    }

    if (!m_paused && !playable) {
        handleRead(); // read more from the source if less than the max buffer
    }
}

void MediaPlayer::onSourceTrack(int track, std::shared_ptr<MediaFormat> format)
{
    if (format) {
        const MediaType& type = format->getType();
        m_log.debug("add track %s - %s", media::fourccString(track).c_str(), type.name.c_str());
        m_sink->configure(track, format);

        if (m_playhead.isSeeking() && m_source.isPassthrough()) {
            m_sink->seekTo(m_playhead.getPosition());
        }
    }
}

void MediaPlayer::onSourceSessionData(const std::map<std::string, std::string>& properties)
{
    if (m_sessionData.getProperties().empty()) {
        m_sessionData.reset(properties);
        m_liveStatistics.generateServerOffset(m_sessionData.getServerTime());
    } else {
        m_sessionData.update(properties);
    }
    m_listener.onSessionData(m_sessionData.getProperties());
}

void MediaPlayer::onSourceOpened()
{
    m_log.info("source opened");
    m_source.onOpened();
    MediaSource* source = m_source.getCurrentSource();
    if (!source) {
        return;
    }

    m_qualities.reset(*m_platform, source->getQualities());
    if (m_autoSwitchQuality && !m_qualities.getAutoQualities().empty()) {
        updateAdaptiveQuality();
    } else {
        updateSourceQuality(m_qualities.getSelected());
    }

    // this handles seek() called before the ready state was entered
    if (m_playhead.isSeeking() && source->isSeekable()) {
        handleSeek(m_playhead.getPosition(), true);
    }

    if (m_state != Player::State::Playing && (!m_buffer.isSynchronizedLatencyMode() || m_state != Player::State::Buffering)) {
        // ignore the state change in low latency buffering mode
        updateState(Player::State::Ready);
    }

    bool disableLowLatencyABR = m_autoSwitchQuality && !m_platform->getCapabilities().supportsLowLatencyABR;
    bool isLowLatency = m_liveLowLatencyEnabled && m_sessionData.isLowLatency() && !disableLowLatencyABR;
    // TEMP code for low latency transcodes for Amazon Trivia and 2s sync demo channels
    m_buffer.setSynchronizedLatencyMode(m_sessionData.isUltraLowLatency());

    updateBufferOptions(isLowLatency);
    source->setLowLatencyEnabled(isLowLatency);
    m_qualitySelector.setLowLatencyMode(isLowLatency);

    if (!m_paused) {
        scheduleRead();
    }
}

void MediaPlayer::onSourceSample(int track, std::shared_ptr<MediaSampleBuffer> sample)
{
    m_source.onSample(track, sample);
    m_sink->enqueue(track, sample);
    // update the end time of the buffer range
    m_buffer.updateBufferEnd(track, *sample);
}

void MediaPlayer::onSourceUpdateQuality()
{
    updateAdaptiveQuality();
}

void MediaPlayer::onSinkDurationChanged(MediaTime duration)
{
    onSourceDurationChanged(duration);
}

void MediaPlayer::onSinkError(const Error& error)
{
    handleError(error);
}

void MediaPlayer::onSinkFormatChanged(const MediaFormat& format)
{
    const auto& type = format.getType();
    if (!type.isText()) {
        checkQualityChanged(format.getName());
    }

    if (format.getPath() != m_source.getPath()) {
        m_source.onPlaying(format.getPath());
    }
}

void MediaPlayer::onSinkIdle()
{
    if (m_paused || m_state == Player::State::Idle) {
        return;
    }

    if (m_source.isEnded()) {
        if (m_looping && m_source.isSeekable()) {
            handleSeek(MediaTime::zero(), true);
        } else {
            m_sink->pause();
            m_log.info("end of stream");
            updateState(Player::State::Ended);
            m_cancelRead.cancel(); // no need to fill buffer in this case
        }

    } else if (m_state != Player::State::Buffering) {

        if (!m_source.isPassthrough()) { // Don't pause passthrough sink
            m_sink->pause();
        }

        if (m_state == Player::State::Playing && !m_playhead.isSeeking()) {
            m_log.warn("rebuffering... position: %lld", m_playhead.getPosition().microseconds().count());
            m_listener.onRebuffering();
            updateBufferRange(TimeRange());
            updateBufferState(BufferState::Refilling);
            // in low latency mode reloads the source to catch up back to live
            if (m_buffer.isSynchronizedLatencyMode() && m_source.isLive()) {
                handleClose(false, Player::State::Buffering);
            }
            updateAdaptiveQuality(); // allows the quality to be switched down on-rebuffering
            handleRead();
        }

        updateState(Player::State::Buffering);
    }
}

void MediaPlayer::onSinkMetadataSample(const MediaSampleBuffer& sample)
{
    if (sample.type == media::fourcc("json")) {
        // twitch specific json metadata
        m_metadataParser.onSample(sample, *this);
        m_listener.onMetadata(MediaType::Text_Json.name, sample.buffer);
    } else if (sample.type == media::fourcc("wvtt")) {
        std::string text(sample.buffer.data(), sample.buffer.data() + sample.buffer.size());
        auto caption = json11::Json::object { { "caption",
            json11::Json::object {
                { "text", text },
                { "timestamp", sample.decodeTime.seconds() },
            } } };
        auto json = json11::Json(caption).dump();
        std::vector<uint8_t> buffer(json.begin(), json.end());
        m_listener.onMetadata(MediaType::Text_Json.name, buffer);
    } else if (sample.type == media::fourcc("PRIV")) {
        // ID3 priv binary
        m_listener.onMetadata(MediaType::Application_OctetStream.name, sample.buffer);
    } else {
        m_log.warn("ignored metadata sample %d", sample.type);
    }
}

void MediaPlayer::onSinkRecoverableError(const Error& error)
{
    m_listener.onRecoverableError(error);
}

void MediaPlayer::onSinkStateChanged(MediaSink::State state)
{
    switch (state) {
    case MediaSink::State::Playing:
        updateState(Player::State::Playing);
        break;
    case MediaSink::State::Idle:
        onSinkIdle();
        break;
    case MediaSink::State::Ended:
        // passthrough only
        onSourceEndOfStream();
        break;
    }
}

void MediaPlayer::onSinkTimeUpdate(MediaTime time)
{
    MediaTime current = m_playhead.getPosition();
    if (m_playhead.setPosition(time) && !m_paused) {
        // read more data from the source if the playhead interval has passed
        scheduleRead();
    }

    if (current != time) {
        m_listener.onPositionChanged(m_playhead.getPosition());
    }
}

void MediaPlayer::onSinkVideoStatistics(const Statistics& statistics)
{
    // update player statistics
    m_statistics.copy(statistics);
    // TODO schedule in native PlaybackSink
    schedule([this]() {
        if (m_autoSwitchQuality) {
            m_qualitySelector.onStatistics(m_statistics, m_qualities.getCurrent());
        }
    });
}

void MediaPlayer::onMetaServerAdBreakStart(MetadataParser::AdRollType rollType)
{
    (void)rollType;
    m_resumable = true;
}

void MediaPlayer::onMetaServerAdBreakEnd()
{
    MediaSource* source = m_source.getCurrentSource();
    if (m_resumable && source && source->isLive()) {
        // check if we need to jump back to live
        m_resumable = false;
        if (m_metadataParser.getAdRollType() == MetadataParser::AdRollType::Pre) {
            auto range = m_buffer.getPlayableRange(m_playhead.getPosition());
            // Use ~1 segment more than min buffer
            MediaTime extraAdBuffer = m_sessionData.isLowLatency() ? MediaTime::zero() : MediaTime(2.0);
            MediaTime seek = range.end() - (m_buffer.getMinBuffer() + extraAdBuffer);
            if (seek > range.start) {
                m_sink->seekTo(seek);
            }
        }
    }
}

void MediaPlayer::onMetaAdBreakRequested(MediaTime duration)
{
    (void)duration;
}

void MediaPlayer::onMetaLatencyTiming(MediaTime streamOffset, MediaTime receiveTime, MediaTime sendTime)
{
    if (!m_resumable && !m_paused) {
        m_liveStatistics.update(streamOffset, receiveTime, sendTime);
    }
}

void MediaPlayer::onMetaSessionData(const std::map<std::string, std::string>& values)
{
    m_sessionData.update(values);
    m_listener.onSessionData(m_sessionData.getProperties());
}

bool MediaPlayer::checkPlayable()
{
    const TimeRange range = m_buffer.getPlayableRange(m_playhead.getPosition());
    // check if we have min buffered or if we're at the end of the stream
    MediaTime minBuffer = m_buffer.getMinBuffer();
    MediaTime duration = m_source.getDuration();

    if (m_source.isPassthrough()
        || range.duration >= minBuffer
        || (duration > MediaTime::zero() && duration - range.start <= minBuffer)
        || m_source.isEnded()) {

        updateBufferRange(range);

        // low latency mode check (buffering time too long) if so reload the source
        if (m_source.isLive() && m_buffer.isBufferingTimedOut(m_liveStatistics.getBroadcasterLatency())) {
            m_log.warn("buffering time %lld us above threshold", m_buffer.getFillTime().microseconds().count());
            handleClose(true, Player::State::Buffering);
            handleRead();
            return false;
        }

        if (m_playhead.isSeeking()) {
            m_log.info("seeked to %lld us", m_playhead.getPosition().microseconds().count());
            m_listener.onSeekCompleted(m_playhead.getPosition());
            m_playhead.seekCompleted();
        }

        if (m_paused) {
            m_sink->prepare();
            if (m_state == Player::State::Buffering) {
                updateState(Player::State::Idle);
            }
        } else {
            m_sink->play();
        }

        updateBufferState(BufferState::Draining);
        return true;
    }

    return false;
}

void MediaPlayer::checkBufferSpeedUp()
{
    if (m_platform->getCapabilities().supportsSpeedUp && m_state == Player::State::Playing && m_source.isLive()) {
        // Check for speedup if in low latency mode
        float rate = m_resumable ? 1.0f : m_buffer.getSpeedUpRate(m_playhead.getPosition());
        if (rate != m_playhead.getPlaybackRate()) {
            m_playhead.setPlaybackRate(rate);
            m_sink->setPlaybackRate(rate);
            m_log.info("Set speedup playback rate %f", rate);
        }
    }
}

void MediaPlayer::checkQualityChanged(const std::string& name)
{
    // the format name will be set to quality it came from, use that to match the quality
    if (!name.empty() && name != m_qualities.getCurrent().name) {
        for (const auto& quality : m_qualities.getQualities()) {
            if (quality.name == name) {
                std::string current = m_qualities.getCurrent().name;
                m_log.info("quality changed to %s from %s",
                    quality.name.c_str(), current.empty() ? "none" : current.c_str());
                m_qualities.setCurrent(quality);
                m_listener.onQualityChanged(quality);
                break;
            }
        }
    }
}

bool MediaPlayer::switchNextQuality(bool reset)
{
    // try to switch the source to the next lowest quality, return false if there is nothing to switch to
    if (m_source.isReadable() && m_qualities.getAutoQualities().size() > 1) {
        const auto& current = (m_qualities.getCurrent() == Quality()) ? m_qualities.getSelected() : m_qualities.getCurrent();
        m_qualities.remove(current, false);
        // match should return next lowest
        Quality quality = m_qualities.match(current.bitrate);

        if (!current.name.empty() && quality != current) {
            m_log.warn("downgrade quality to %s from %s", quality.name.c_str(), current.name.c_str());
            if (reset) { // reset pipeline
                updateSourceQuality(quality);
                m_sink->reset();
                m_sink = createSink();
                // allows the source to resume spliced at the current time
                handleSeekToDefault();
                return true;
            } else {
                const auto& currentSelected = m_qualities.getSelected();
                updateAdaptiveQuality();
                return m_qualities.getSelected() != currentSelected;
            }
        }
    }
    return false;
}

bool MediaPlayer::updateAdaptiveQuality()
{
    // update the abr algorithm with the current state before selecting the next quality
    MediaSource* source = m_source.getCurrentSource();
    // in chunked streaming mode the buffer size will be smaller, so need to be more aggressive switching up
    if (source && (m_sessionData.isLowLatency() || m_platform->getName() == "web")) {
        m_qualitySelector.setTargetBufferSize(m_buffer.getMinBuffer());
    }

    updateBufferRange(m_buffer.getPlayableRange(m_playhead.getPosition()));

    if (m_autoSwitchQuality && (source && !source->isPassthrough())) {
        const auto& quality = m_qualitySelector.nextQuality(m_qualities);

        if (m_qualities.getSelected().name != quality.name) {
            Quality target = m_qualities.match(quality);
            m_qualities.setSelected(target);
            m_log.debug("adaptive set quality to %s (%d)", quality.name.c_str(), quality.bitrate);
            m_source.setQuality(m_qualities.getSelected(), true);
            return true;
        }
    }
    return false;
}

void MediaPlayer::updateBufferRange(const TimeRange& range)
{
    m_qualitySelector.onBufferDurationChange(range);
}

void MediaPlayer::updateBufferState(BufferState state)
{
    m_buffer.setState(state);
    m_qualitySelector.onBufferStateChange(state);
}

void MediaPlayer::updateSourceQuality(const Quality& quality)
{
    m_log.debug("set quality to %s (%d)", quality.name.c_str(), quality.bitrate);
    if (!m_qualities.isEmpty()) {
        Quality target = m_qualities.match(quality);
        m_qualities.setCurrent(Quality());
        m_qualities.setSelected(target);
        m_source.setQuality(m_qualities.getSelected(), false);
    } else {
        m_qualities.setSelected(quality);
    }
}

void MediaPlayer::handleError(const Error& error)
{
    if (error.result != MediaResult::Ok) {
        m_log.warn("error %s:%d (%s code %d - %s)",
            errorSourceString(error.source), error.result.value,
            mediaResultString(error.result), error.result.code, error.message.c_str());

        bool propagate = true;
        // try switching to another quality if not possible error
        if (!m_qualities.getAutoQualities().empty()) {
            if ((error.source == ErrorSource::Segment || error.source == ErrorSource::Playlist)
                && (error.result.code == HttpStatusNotFound || error.result.code == HttpStatusGone)) {
                // try to switch to the next quality preserving the current player buffer
                propagate = !switchNextQuality(false);
            } else if ((error.source == ErrorSource::Decode || error.source == ErrorSource::Render)
                && error.result != MediaResult::ErrorNetwork) {
                // try to switch to the next quality and also reset the media sink(s)
                propagate = !switchNextQuality(true);
            }
        }

        if (propagate) {
            m_log.warn("error stopping playback");
            m_sink.reset();
            m_sink = createSink();
            m_paused = true;
            m_remote = false;
            handleClose(true);
            m_listener.onError(error);
        } else {
            m_listener.onRecoverableError(error);
        }
    }
}

void MediaPlayer::handleOpen(const std::string& path, const MediaType& mediaType)
{
    // Reset data that may have referred to previously played content
    m_statistics.reset();
    m_liveStatistics = LatencyStatistics();
    m_buffer.reset();
    m_qualities.clear();
    m_qualitySelector.onStreamChange();
    m_sessionData.clear();
    m_listener.onSessionData(m_sessionData.getProperties());

    if (m_sink) {
        m_sink->seekTo(MediaTime::zero());
    }

    auto source = createSource(path, mediaType, m_remote);
    if (!source) {
        handleError(Error(ErrorSource::Source, MediaResult::ErrorNoSource, "Source create failed"));
    } else {
        m_source.clear(); // only supporting one source at time currently
        m_source.add(path, std::move(source));
        m_source.open();
    }
}

void MediaPlayer::handleClose(bool reset, Player::State state)
{
    bool canResume = !m_source.isLive() || m_resumable;
    m_cancelRead.cancel();

    if (!canResume) {
        m_source.close();
    }

    if (m_sink) {
        m_sink->pause();
        updateState(state);

        if (reset) {
            m_sink->reset();
            m_buffer.reset();
            m_playhead.reset();
        } else if (!canResume) {
            handleSeek(MediaTime::zero(), false);
        }
    }
}

void MediaPlayer::handleSeek(MediaTime time, bool read, bool remove)
{
    m_sink->pause();
    if (remove) {
        // this clears the buffer contents
        m_sink->remove(TimeRange(MediaTime::zero(), MediaTime::max()));
    }
    m_sink->seekTo(time);

    m_playhead.reset();
    m_playhead.seekTo(time);
    m_buffer.reset();
    m_buffer.seekTo(time);

    if (read) {
        if (m_source.isReadable()) {
            m_source.seekTo(time);
        }
        // there's a delay here to debounce a large amount of seek calls
        scheduleRead(m_playhead.getInterval());
        updateState(Player::State::Buffering);
    }
}

void MediaPlayer::handleSeekToDefault()
{
    if (!m_source.isPassthrough()) {
        if (m_source.isLive()) {
            // TODO in the future would seek inside live absolute timeline
            handleSeek(MediaTime::zero(), !m_paused);
        } else {
            handleSeek(m_playhead.getPosition(), true);
        }
    }
}

void MediaPlayer::handleRead()
{
    MediaTime position = m_playhead.getPosition();
    m_buffer.updatePosition(position);
    TimeRange range = m_buffer.getPlayableRange(position);

    if (range.duration < m_buffer.getMaxBuffer()) {
        m_source.read(TimeRange(range.end(), m_buffer.getMinBuffer()));
    }

    // trim buffer data before the playable range, only 1 buffered range is supported
    if (m_state == Player::State::Playing) {
        TimeRange discard = m_buffer.getBufferTrimRange(range.start);
        if (discard.duration > MediaTime::zero()) {
            m_sink->remove(discard);
        }
    }

    // update the buffer state if needed
    BufferState bufferState = m_buffer.getState();
    if (bufferState != BufferState::Draining
        && bufferState != BufferState::Filling
        && bufferState != BufferState::Refilling) {
        m_buffer.setState(BufferState::Filling);
        m_qualitySelector.onBufferStateChange(m_buffer.getState());
    }
    updateBufferRange(range);
    checkBufferSpeedUp();
}

void MediaPlayer::removeQuality(const Quality& quality)
{
    m_qualities.remove(quality);
    if (m_qualities.isEmpty()) {
        handleError(Error(ErrorSource::Source, MediaResult::Error, "No playable quality"));
    }
}

void MediaPlayer::setHidden(bool hidden)
{
    if (m_hidden != hidden) {
        m_hidden = hidden;
        // TODO background mode should use audio only
        const int BackgroundBitrate = 1500000;
        const int MaxBitrate = std::numeric_limits<int>::max();

        if (m_autoSwitchQuality) {
            setAutoMaxBitrate(hidden ? BackgroundBitrate : MaxBitrate);
        } else {
            Quality backgroundQuality = m_qualities.match(BackgroundBitrate);
            if (hidden) {
                if (backgroundQuality.bitrate < m_qualities.getSelected().bitrate) {
                    setQuality(backgroundQuality, true);
                    m_changedVisibleQuality = true;
                }
            } else {
                if (m_changedVisibleQuality && m_qualities.getPrevious().bitrate >= backgroundQuality.bitrate) {
                    setQuality(m_qualities.getPrevious(), true);
                    m_changedVisibleQuality = false;
                }
            }
        }

        if (!hidden) {
            replaceBuffer(true);
        }
    }
}

void MediaPlayer::replaceBuffer(bool nonAdaptive)
{
    if (m_state == State::Playing
        && !m_resumable && m_platform->getCapabilities().supportsBufferOverwrite) {
        // check if it's actually worth replacing the buffer
        MediaTime position = m_playhead.getPosition();
        MediaTime start = position + m_buffer.getMinBuffer();
        MediaTime duration = m_buffer.getBufferEnd() - start;
        // ensure there is enough buffered and there is an sync frame ahead of the play head
        MediaTime nextSync = m_buffer.getSyncTimeBetween(position, MediaTime::max());
        if (start < m_buffer.getBufferEnd() && nextSync != MediaTime::invalid()) {
            bool seekSource = false;
            if (m_autoSwitchQuality) {
                if (m_qualitySelector.canReplaceBuffer(m_qualities, duration)) {
                    if (updateAdaptiveQuality()) { // check if source quality actually changed
                        seekSource = true;
                    }
                }
            } else if (nonAdaptive) {
                // manual mode case assume user has enough bw for now
                seekSource = true;
            }
            if (seekSource) {
                m_log.debug("replace buffer at %.3f position %.3f", nextSync.seconds(), position.seconds());
                m_source.seekTo(nextSync);
            }
        }
    }
}

void MediaPlayer::updateBufferOptions(bool isLowLatency)
{
    m_buffer.update(m_settings, isLowLatency);
}

std::shared_ptr<Player> Player::create(Player::Listener& listener, std::shared_ptr<Platform> platform)
{
    return std::make_shared<MediaPlayer>(listener, platform);
}

const char* Player::stateToString(Player::State state)
{
    const char* names[] = {
        "Idle",
        "Ready",
        "Buffering",
        "Playing",
        "Ended"
    };
    return names[static_cast<int>(state)];
}
}
