#pragma once

#include "BufferControl.hpp"
#include "CompositeListener.hpp"
#include "MultiSource.hpp"
#include "Playhead.hpp"
#include "Qualities.hpp"
#include "ScopedScheduler.hpp"
#include "Settings.hpp"
#include "VideoStatistics.hpp"
#include "abr/QualitySelector.hpp"
#include "analytics/AnalyticsTracker.hpp"
#include "cancellation.hpp"
#include "debug/PrefixLog.hpp"
#include "media/Resolution.hpp"
#include "playercore/MediaSink.hpp"
#include "playercore/MediaSource.hpp"
#include "playercore/Player.hpp"
#include "playercore/platform/Platform.hpp"
#include "twitch/LatencyStatistics.hpp"
#include "twitch/MetadataParser.hpp"
#include "twitch/SessionData.hpp"
#include "twitch/TokenHandler.hpp"

namespace twitch {
class MediaPlayer : public Player,
                    protected ScopedScheduler,
                    private MediaSource::Listener,
                    private MediaSink::Listener,
                    private MetadataParser::Listener {
public:
    MediaPlayer(Player::Listener& listener, std::shared_ptr<Platform> platform);
    ~MediaPlayer() override;
    MediaPlayer(const MediaPlayer&) = delete;
    const MediaPlayer& operator=(const MediaPlayer&) = delete;

    void load(const std::string& path) override;
    void load(const std::string& path, const std::string& mediaType) override;
    void play() override;
    void pause() override;
    void seekTo(MediaTime time) override;
    bool isSeekable() const override;
    MediaTime getDuration() const override;
    MediaTime getPosition() const override;
    MediaTime getBufferedPosition() const override;
    Player::State getState() const override { return m_state; }
    const std::string& getName() const override;
    const std::string& getVersion() const override;
    bool isLooping() const override { return m_looping; }
    void setLooping(bool loop) override { m_looping = loop; }
    bool isMuted() const override { return m_muted; }
    void setMuted(bool muted) override;
    float getVolume() const override { return m_volume; }
    void setVolume(float volume) override;
    const Quality& getQuality() const override;
    void setQuality(const Quality& quality) override;
    void setQuality(const Quality& quality, bool adaptive) override;
    const std::vector<Quality>& getQualities() const override;
    bool getAutoSwitchQuality() const override;
    void setAutoSwitchQuality(bool enable) override;
    int getAverageBitrate() const override;
    int getBandwidthEstimate() const override;
    MediaTime getLiveLatency() const override;
    float getPlaybackRate() const override;
    void setPlaybackRate(float rate) override;
    void setSurface(void* surface) override;
    std::string getPath() const override;
    void setClientId(const std::string& id) override;
    void setDeviceId(const std::string& id) override;
    void setAuthToken(const std::string& token) override;
    void setPlayerType(const std::string& type) override;
    void setAutoInitialBitrate(int bitrate) override;
    void setAutoMaxBitrate(int bitrate) override;
    void setAutoMaxVideoSize(int width, int height) override;
    void setAutoViewportSize(int width, int height);
    void setAutoViewportScale(float scale);
    void setLiveMaxLatency(MediaTime time) override;
    void setLiveLowLatencyEnabled(bool enable) override;
    void setMinBuffer(MediaTime duration) override;
    void setMaxBuffer(MediaTime duration) override;
    void startRemotePlayback();
    void endRemotePlayback();
    std::shared_ptr<Platform> getPlatform() const;
    // internal only
    void setAnalyticsSendEvents(bool enable);
    void setAnalyticsEndpoint(const std::string& endpoint);
    const Statistics& getStatistics() const override { return m_statistics; }

    // for web only
    void setLiveSpeedUpRate(float rate);
    void setSettings(const json11::Json& json);
    void setVisible(bool visible);
    MediaSink& getSink() const { return *m_sink; }
    const abr::QualitySelector& getQualitySelector() const { return m_qualitySelector; }
    void removeQuality(const Quality& quality);
    void setRequestListener(MediaSource::Request::Listener* listener) { m_requestListener = listener; }
    const LatencyStatistics& getLatencyStatistics() const { return m_liveStatistics; }

    // for tv-apps
    void requestServerAd() override;
    void regenerateAnalyticsPlaySession() override;
    void setKeepAnalyticsPlaySession(bool keepPlaySession) override;

protected:
    void onRequestSent(const MediaSource::Request& request) override;
    void onResponseReceived(const MediaSource::Request& request) override;
    void onResponseBytes(const MediaSource::Request& request, size_t bytes) override;
    void onResponseEnd(const MediaSource::Request& request) override;
    void onRequestError(const MediaSource::Request& request, int error) override;
    void onSinkDurationChanged(MediaTime time) override;
    void onSinkError(const Error& error) override;
    void onSinkFormatChanged(const MediaFormat& format) override;
    void onSinkIdle();
    void onSinkMetadataSample(const MediaSampleBuffer& sample) override;
    void onSinkRecoverableError(const Error& error) override;
    void onSinkStateChanged(MediaSink::State state) override;
    void onSinkTimeUpdate(MediaTime time) override;
    void onSourceDurationChanged(MediaTime duration) override;
    void onSinkVideoStatistics(const Statistics& statistics) override;
    void onSourceSeekableChanged(bool seekable) override;
    void onSourceEndOfStream() override;
    void onSourceError(const Error& error) override;
    void onSourceRecoverableError(const Error& error) override;
    void onSourceFlush() override;
    void onSourceOpened() override;
    void onSourceSample(int track, std::shared_ptr<MediaSampleBuffer> sample) override;
    void onSourceTrack(int track, std::shared_ptr<MediaFormat> format) override;
    void onSourceSessionData(const std::map<std::string, std::string>& properties) override;
    void onSourceUpdateQuality() override;
    void onMetaServerAdBreakStart(MetadataParser::AdRollType rollType) override;
    void onMetaServerAdBreakEnd() override;
    void onMetaAdBreakRequested(MediaTime duration) override;
    void onMetaLatencyTiming(MediaTime streamOffset, MediaTime receiveTime, MediaTime sendTime) override;
    void onMetaSessionData(const std::map<std::string, std::string>& values) override;

private:
    std::unique_ptr<MediaSource> createSource(const std::string& path, const MediaType& mediaType, bool remote = false);
    std::unique_ptr<MediaSink> createSink();
    void checkBufferSpeedUp();
    void checkQualityChanged(const std::string& name);
    bool checkPlayable();
    bool switchNextQuality(bool reset);
    bool updateAdaptiveQuality();
    void updateBufferRange(const TimeRange& range);
    void updateBufferState(BufferState state);
    void updateSourceQuality(const Quality& quality);
    void updateState(Player::State state);
    void scheduleRead(MediaTime delay = MediaTime::zero());
    void handleOpen(const std::string& path, const MediaType& mediaType);
    void handleClose(bool reset, Player::State state = Player::State::Idle);
    void handleSeek(MediaTime time, bool read, bool remove = true);
    void handleSeekToDefault();
    void handleRead();
    void handleError(const Error& error);
    void setHidden(bool visible);
    void replaceBuffer(bool nonAdaptive);
    void updateBufferOptions(bool isLowLatency);

    std::unique_ptr<analytics::AnalyticsTracker> m_analytics;
    CompositeListener m_listener;
    MediaSource::Request::Listener* m_requestListener;
    MultiSource m_source;
    std::unique_ptr<MediaSink> m_sink;
    std::shared_ptr<Platform> m_platform;
    std::shared_ptr<TokenHandler> m_tokenHandler;
    float m_volume;
    bool m_muted;
    std::string m_path;
    MediaType m_mediaType;
    Player::State m_state;
    BufferControl m_buffer;
    Playhead m_playhead;
    PrefixedLog m_log;
    CancellableRef m_cancelRead;
    CancellableRef m_cancelHidden;
    bool m_looping;
    bool m_paused;
    bool m_autoSwitchQuality;
    bool m_resumable;
    bool m_remote;
    Qualities m_qualities;
    abr::QualitySelector m_qualitySelector;
    Settings m_settings;
    VideoStatistics m_statistics;
    bool m_liveLowLatencyEnabled;
    bool m_hidden;
    bool m_changedVisibleQuality;
    LatencyStatistics m_liveStatistics;
    SessionData m_sessionData;
    media::Resolution m_viewportSize;
    void* m_surface;
    MetadataParser m_metadataParser;
};
}
