#include "MediaRequest.hpp"
#include "util/Random.hpp"
#include <cmath>
#include <cstdlib>

namespace twitch {
const std::string MediaRequest::Type::MasterPlaylist = "MasterPlaylist";
const std::string MediaRequest::Type::MediaPlaylist = "MediaPlaylist";
const std::string MediaRequest::Type::Segment = "Segment";

int MediaRequest::generateRequestId()
{
    static int value = 0;
    return value++;
}

MediaRequest::MediaRequest(const std::string& type)
    : m_id(generateRequestId())
    , m_type(type)
    , m_httpStatus(0)
    , m_attempts(0)
    , m_maxAttempts(3)
    , m_networkError(0)
    , m_pending(false)
    , m_complete(false)
    , m_appendedBytes(0)
    , m_receivedBytes(0)
    , m_contentLength(0)
    , m_listener(nullptr)
{
}

bool MediaRequest::isFailed() const
{
    return !m_pending && (m_networkError != 0 || m_httpStatus >= HttpStatusMultipleChoices || m_httpStatus < HttpStatusOk);
}

std::chrono::microseconds MediaRequest::getRetryTime() const
{
    using namespace std::chrono;
    double attempt = std::max(m_attempts, 1);
    const std::chrono::seconds RetryInterval(1);
    return Random::jitter(milliseconds(100), duration_cast<milliseconds>(std::pow(2.0, attempt - 1) * RetryInterval));
}

void MediaRequest::cancel()
{
    m_cancelToken.cancel();
    if (m_request) {
        m_request->cancel();
    }

    m_pending = false;
    m_attempts = 0;
    m_receivedBytes = 0;
    m_appendedBytes = 0;
    m_contentLength = 0;
    m_complete = false;
}

void MediaRequest::appendedBytes(size_t size, bool endOfStream)
{
    m_appendedBytes += size;

    if (endOfStream) {
        m_complete = true;
    }

    if (m_listener) {
        m_listener->onResponseBytes(*this, size);
        if (endOfStream) {
            m_listener->onResponseEnd(*this);
        }
    }
}

size_t MediaRequest::skipBytes(size_t size)
{
    m_receivedBytes += size;

    if (m_httpStatus != HttpStatusPartialContent) { // 206 partial content means range request succeeded
        // if just 200 we are getting the whole segment
        // skip over this data as it was already appended in the previous attempt
        if (size && m_appendedBytes >= m_receivedBytes) {
            return size;
        }

        // if overlapping with already appended data append from the transferred offset
        if (m_receivedBytes - size < m_appendedBytes && m_appendedBytes < m_receivedBytes) {
            size_t offset = m_appendedBytes - (m_receivedBytes - size);
            return offset;
        }
    }

    return 0;
}

void MediaRequest::retry(Scheduler& scheduler, Scheduler::Action action)
{
    setCancellable(scheduler.schedule(std::move(action), getRetryTime()));
}

void MediaRequest::setCancellable(const CancellableRef& cancellable)
{
    m_pending = true;
    m_cancelToken = cancellable;
}

void MediaRequest::onRequest(std::shared_ptr<HttpRequest> request)
{
    m_request = std::move(request);
    m_pending = true;
    m_attempts++;
    m_receivedBytes = 0;

    if (m_appendedBytes && (m_type == Type::Segment || m_type == "File")) {
        // disable gzip on these requests in case it breaks the range request
        m_request->setHeader("Accept-Encoding", "identity");
        m_request->setHeader("Range", "bytes=" + std::to_string(m_appendedBytes) + "-");
    }

    if (m_listener) {
        m_listener->onRequestSent(*this);
    }
}

void MediaRequest::onResponse(const HttpResponse& response)
{
    m_httpStatus = response.getStatus();
    m_contentType = response.getHeader("Content-Type");

    std::string contentLength = response.getHeader("Content-Length");
    if (!contentLength.empty()) {
        m_contentLength = static_cast<size_t>(std::strtoul(contentLength.c_str(), nullptr, 10));
    }

    if (m_listener && response.isSuccess()) {
        m_listener->onResponseReceived(*this);
    }

    if (!response.isSuccess()) {
        m_pending = false;
    }
}

void MediaRequest::onNetworkError(int error)
{
    m_networkError = error;
    m_pending = false;

    if (m_listener) {
        m_listener->onRequestError(*this, error);
    }
}

void MediaRequest::onCompleted()
{
    m_pending = false;
    m_attempts = 0;
    m_appendedBytes = 0;
}

void MediaRequest::readString(HttpResponse& response, StringHandler onString, HttpResponse::ErrorHandler onError)
{
    // note this has to be a shared ptr since the buffer handler std::function is copied
    std::shared_ptr<std::string> content = std::make_shared<std::string>();

    response.read([onString, content](const uint8_t* buffer, size_t size, bool endOfStream) mutable {
        if (buffer) {
            content->insert(content->end(), buffer, buffer + size);
        }

        if (endOfStream) {
            onString(*content);
        }
    },
        std::move(onError));
}
}
