#pragma once

#include "player/cancellation.hpp"
#include "playercore/MediaSource.hpp"
#include "playercore/MediaTime.hpp"
#include "playercore/platform/HttpRequest.hpp"
#include "playercore/platform/Platform.hpp"

namespace twitch {
class MediaRequest : public MediaSource::Request {
public:
    struct Type {
        static const std::string MasterPlaylist;
        static const std::string MediaPlaylist;
        static const std::string Segment;
    };

    explicit MediaRequest(const std::string& type);
    ~MediaRequest() override = default;
    int getId() const override { return m_id; }
    const std::string& getType() const override { return m_type; }
    const std::string& getUrl() const { return m_url; }
    void setUrl(const std::string& url) { m_url = url; }
    const std::string& getContentType() const { return m_contentType; }
    size_t getContentLength() const override { return m_contentLength; }
    std::chrono::microseconds getRetryTime() const;
    bool isComplete() const { return m_complete; }
    bool isPending() const { return m_pending; }
    bool isFailed() const;
    bool isRetryable() const { return m_attempts < m_maxAttempts; }
    bool isMediaPrefetch() const override { return false; }
    bool isRangeRequest() const override { return false; }
    void setMaxAttempts(int attempts) { m_maxAttempts = attempts; }
    void setCancellable(const CancellableRef& cancellable);
    void setListener(MediaSource::Request::Listener* listener) { m_listener = listener; }
    int getMediaBitrate() const override { return 0; }
    MediaTime getMediaDuration() const override { return m_mediaDuration; }
    void setMediaDuration(MediaTime duration) { m_mediaDuration = duration; }
    size_t getAppendedBytes() const { return m_appendedBytes; }
    void appendedBytes(size_t size, bool endOfStream);
    size_t skipBytes(size_t size);

    using StringHandler = std::function<void(const std::string& content)>;
    void readString(HttpResponse& response, StringHandler onString, HttpResponse::ErrorHandler onError);

    void retry(Scheduler& scheduler, Scheduler::Action action);

    virtual void cancel();
    virtual void onRequest(std::shared_ptr<HttpRequest> request);
    virtual void onResponse(const HttpResponse& response);
    virtual void onNetworkError(int error);
    virtual void onCompleted();

protected:
    static int generateRequestId();

    int m_id;
    std::string m_url;
    std::string m_type;
    int m_httpStatus;
    int m_attempts;
    int m_maxAttempts;
    int m_networkError;
    bool m_pending;
    bool m_complete;
    size_t m_appendedBytes;
    size_t m_receivedBytes;
    size_t m_contentLength;
    std::string m_contentType;
    std::shared_ptr<HttpRequest> m_request;
    CancellableRef m_cancelToken;
    MediaSource::Request::Listener* m_listener;
    MediaTime m_mediaDuration;
};
}
