#include "MultiSource.hpp"

namespace twitch {
MultiSource::MultiSource()
    : m_reading(InvalidSourceId)
    , m_playing(InvalidSourceId)
{
}

MultiSource::~MultiSource()
{
    clear();
}

void MultiSource::add(const std::string& path, std::unique_ptr<MediaSource> source, MediaTime playDuration)
{
    if (!source) {
        return;
    }

    int nextId = m_sources.empty() ? 1 : ((--m_sources.end())->first + 1);

    if (m_reading == InvalidSourceId || isEnded()) {
        m_reading = nextId;
    }

    if (m_playing == InvalidSourceId) {
        m_playing = m_reading;
    }

    auto& state = m_sources[nextId];
    state.path = path;
    state.source = std::move(source);
    state.playDuration = playDuration;
}

void MultiSource::open()
{
    if (m_reading != InvalidSourceId) {
        auto& state = m_sources[m_reading];
        if (state.status == SourceState::Status::Opened || state.ended) {
            return;
        }

        state.source->open();
        // handles the case of onSourceOpened called in open() itself
        if (state.status != SourceState::Status::Opened) {
            state.status = SourceState::Status::Opening;
        }
    }
}

void MultiSource::close()
{
    if (m_reading != InvalidSourceId) {
        auto& state = m_sources[m_reading];

        if (state.status != SourceState::Status::Closed) {
            // only closes the source if it's not resumable
            bool canResume = !state.source->isLive();
            if (!canResume) {
                state.source->close();
                state.status = SourceState::Status::Closed;
                return;
            } else {
                state.status = SourceState::Status::Opened; // source remains open
            }
        }
    }

    m_sourceOffset = MediaTime::zero();
}

void MultiSource::clear()
{
    m_sources.clear();
    m_reading = InvalidSourceId;
    m_playing = InvalidSourceId;
}

void MultiSource::read(const TimeRange& range)
{
    if (m_reading != InvalidSourceId) {
        const auto& state = m_sources[m_reading];

        if (state.status != SourceState::Status::Opened && state.status != SourceState::Status::Opening) {
            open();
            return;
        }

        if (state.status == SourceState::Status::Opened) {
            state.source->read(range);
        }
    }
}

void MultiSource::seekTo(MediaTime time)
{
    if (m_reading != InvalidSourceId) {
        auto& state = m_sources[m_reading];

        if (state.status == SourceState::Status::Opened) {
            state.ended = false;
            state.source->seekTo(time);
        }
    }
}

bool MultiSource::isEnded()
{
    auto it = m_sources.find(m_reading);
    if (it != m_sources.end()) {
        return it->second.ended && m_sources.find(m_reading + 1) == m_sources.end();
    }
    return false;
}

bool MultiSource::isLive() const
{
    auto it = m_sources.find(m_playing);
    return it != m_sources.end() ? it->second.live : false;
}

bool MultiSource::isReadable() const
{
    auto it = m_sources.find(m_reading);
    return it != m_sources.end() ? (it->second.status == SourceState::Status::Opened) : false;
}

bool MultiSource::isPassthrough() const
{
    auto it = m_sources.find(m_playing);
    return it != m_sources.end() ? it->second.source->isPassthrough() : false;
}

bool MultiSource::isSeekable() const
{
    auto it = m_sources.find(m_playing);
    return it != m_sources.end() ? it->second.seekable : false;
}

MediaTime MultiSource::getDuration() const
{
    int id = getActiveId();
    auto it = m_sources.find(id);
    if (it != m_sources.end() && it->second.source) {
        return it->second.duration;
    }
    return MediaTime::zero();
}

const std::string& MultiSource::getPath() const
{
    int id = getActiveId();
    auto it = m_sources.find(id);
    if (it != m_sources.end()) {
        return it->second.path;
    }

    static std::string empty;
    return empty;
}

void MultiSource::setQuality(const Quality& quality, bool adaptive)
{
    invokeOnActiveSource(&MediaSource::setQuality, quality, adaptive);
}

const Quality& MultiSource::getQuality() const
{
    MediaSource* source = getCurrentSource();
    if (source) {
        return source->getQuality();
    }
    static Quality empty;
    return empty;
}

const std::vector<Quality>& MultiSource::getQualities() const
{
    int id = getActiveId();

    if (id != InvalidSourceId) {
        auto it = m_sources.find(id);
        if (it != m_sources.end() && it->second.source) {
            return it->second.source->getQualities();
        }
    }

    static std::vector<Quality> empty;
    return empty;
}

void MultiSource::setReadTimeout(MediaTime time)
{
    invokeOnActiveSource(&MediaSource::setReadTimeout, time);
}

void MultiSource::setLowLatencyEnabled(bool enable)
{
    invokeOnActiveSource(&MediaSource::setLowLatencyEnabled, enable);
}

MediaSource* MultiSource::getCurrentSource() const
{
    auto it = m_sources.find(m_reading);
    return it != m_sources.end() ? it->second.source.get() : nullptr;
}

int MultiSource::getActiveId() const
{
    int id = m_playing;

    if (!m_playing) {
        id = m_reading;
    }

    return id;
}

void MultiSource::onPlaying(const std::string& path)
{
    // update the playing source id
    for (const auto& entry : m_sources) {
        if (entry.second.path == path) {
            m_playing = entry.first;
        }
    }
}

void MultiSource::onOpened()
{
    if (m_reading != InvalidSourceId) {
        auto& state = m_sources[m_reading];
        state.status = SourceState::Status::Opened;
        state.duration = state.source->getDuration();
        state.seekable = state.source->isSeekable();
        state.live = state.source->isLive();
        state.ended = false;
    }
}

void MultiSource::onDurationChanged(MediaTime duration)
{
    auto it = m_sources.find(m_reading);
    if (it != m_sources.end()) {
        it->second.duration = duration;
        it->second.live = it->second.source->isLive();
    }
}

void MultiSource::onSample(int track, const std::shared_ptr<MediaSampleBuffer>& sample)
{
    (void)track;

    if (sample->isSyncSample) {
        // update the duration of the media read so far aligned to a sync frame
        auto it = m_sources.find(m_reading);
        if (it != m_sources.end() && it->second.sampleTime < sample->presentationTime) {
            it->second.sampleTime = sample->presentationTime;
        }
    }

    // add the source offset time
    sample->decodeTime += m_sourceOffset;
    sample->presentationTime += m_sourceOffset;
}

void MultiSource::onSeekableChanged(bool seekable)
{
    auto it = m_sources.find(m_reading);
    if (it != m_sources.end()) {
        it->second.seekable = seekable;
    }
}

void MultiSource::onEndOfStream(MediaTime end)
{
    auto it = m_sources.find(m_reading);
    if (it != m_sources.end()) {
        it->second.ended = true;

        // switch to reading the next source (if exists) otherwise move the player to ended
        int nextId = m_reading + 1;
        if (m_sources.find(nextId) != m_sources.end()) {
            m_reading = nextId;
            m_sourceOffset += end;
        }
    }
}

void MultiSource::onFlush()
{
    // check if we should stop reading from this source and switch to another
    auto it = m_sources.find(m_reading);
    if (it != m_sources.end()) {
        if (it->second.sampleTime >= it->second.playDuration) {
            it->second.source->close();
            it->second.status = SourceState::Status::Closed;
            // switch to reading the next source
            onEndOfStream(it->second.sampleTime);
        }
    }
}
}
