#pragma once

#include "playercore/MediaSource.hpp"
#include <map>

namespace twitch {
class MultiSource : public MediaSource {
public:
    MultiSource();
    ~MultiSource() override;
    MultiSource(const MultiSource&) = delete;
    const MultiSource& operator=(const MultiSource&) = delete;

    void add(const std::string& path, std::unique_ptr<MediaSource> source, MediaTime playDuration = MediaTime::max());
    void open() override;
    void close() override;
    void clear();
    void read(const TimeRange& range) override;
    void seekTo(MediaTime time) override;
    bool isEnded();
    bool isLive() const override;
    bool isReadable() const;
    bool isPassthrough() const override;
    bool isSeekable() const override;
    MediaTime getDuration() const override;
    const std::string& getPath() const;
    void setQuality(const Quality& quality, bool adaptive) override;
    const Quality& getQuality() const override;
    const std::vector<Quality>& getQualities() const override;
    void setReadTimeout(MediaTime time) override;
    void setLowLatencyEnabled(bool enable) override;
    MediaSource* getCurrentSource() const;
    // source state change callbacks
    void onPlaying(const std::string& path);
    void onOpened();
    void onDurationChanged(MediaTime duration);
    void onSample(int track, const std::shared_ptr<MediaSampleBuffer>& sample);
    void onSeekableChanged(bool seekable);
    void onEndOfStream(MediaTime end);
    void onFlush();

private:
    const int InvalidSourceId = -1;
    int getActiveId() const;

    template <typename Method, typename... Args>
    void invokeOnActiveSource(Method method, Args&&... args)
    {
        int id = getActiveId();
        if (id != InvalidSourceId) {
            const auto& state = m_sources[id];
            if (state.source) {
                (*state.source.*method)(std::forward<Args>(args)...);
            }
        }
    }

    int m_reading;
    int m_playing;

    struct SourceState {
        std::unique_ptr<MediaSource> source;
        enum class Status {
            Opening,
            Opened,
            Closing,
            Closed
        };
        Status status = Status::Closed;
        bool ended = false;
        bool seekable = false;
        bool live = false;
        std::string path;
        std::vector<Quality> qualities;
        MediaTime duration = MediaTime::zero();
        MediaTime playDuration = MediaTime::max();
        MediaTime sampleTime = MediaTime::zero();
    };
    std::map<int, SourceState> m_sources;
    MediaTime m_sourceOffset;
};
}
