#include "playercore/platform/NativePlatform.hpp"
#include "AsyncHttpClient.hpp"
#include "file/DownloadSource.hpp"
#include "file/FileSource.hpp"
#include "media/FileReaderFactory.hpp"
#include "sink/PlaybackSink.hpp"
#include "sink/ThreadScheduler.hpp"
#include <cstdlib>

namespace twitch {
namespace {
NativePlatform::LogHandler g_logHandler;
}

std::shared_ptr<HttpClient> NativePlatform::createAsyncHttpClient(std::shared_ptr<Scheduler> scheduler)
{
    return std::make_shared<AsyncHttpClient>(*this, scheduler);
}

std::unique_ptr<DrmSession> NativePlatform::createDrmSession(const std::vector<uint8_t>& system,
    DrmSession::Listener& listener)
{
    (void)system;
    (void)listener;
    return std::unique_ptr<DrmSession>();
}

std::shared_ptr<Scheduler> NativePlatform::createScheduler(const std::string& name)
{
    return std::make_shared<ThreadScheduler>(*this, name);
}

std::unique_ptr<MediaSink> NativePlatform::createSink(MediaSink::Listener& listener,
    std::shared_ptr<Scheduler> scheduler)
{
    return std::unique_ptr<PlaybackSink>(new PlaybackSink(*this, listener, scheduler));
}

std::unique_ptr<MediaSource> NativePlatform::createSource(const std::string& path, const MediaType& mediaType,
    MediaSource::Listener& listener,
    std::shared_ptr<Scheduler> scheduler)
{
    // special string used for crash testing
    if (path == "abort") {
        std::abort();
    }
    if (MediaType::Video_MP4.matches(mediaType) || MediaType::Video_MP2T.matches(mediaType)) {
        if (path.find("://") != std::string::npos) {
            return std::unique_ptr<MediaSource>(new DownloadSource(listener, *this, scheduler, path));
        } else {
            return std::unique_ptr<MediaSource>(new FileSource(listener, *this, path));
        }
    }
    return nullptr;
}

VideoDecoderCapabilities NativePlatform::getVideoDecoderCapabilities(const MediaType& mediaType)
{
    (void)mediaType;
    return VideoDecoderCapabilities();
}

const std::set<MediaType>& NativePlatform::getSupportedMediaTypes() const
{
    static const std::set<MediaType> types = { MediaType::Video_AVC, MediaType::Audio_AAC };
    return types;
}

const std::set<std::vector<uint8_t>>& NativePlatform::getSupportedProtectionSystems() const
{
    static const std::set<std::vector<uint8_t>> empty;
    return empty;
}

const std::map<std::string, std::string>& NativePlatform::getAnalyticsProperties()
{
    static const std::map<std::string, std::string> empty {};
    return empty;
}

std::shared_ptr<MediaReaderFactory> NativePlatform::getMediaReaderFactory() const
{
    static std::shared_ptr<MediaReaderFactory> factory = std::make_shared<media::FileReaderFactory>();
    return factory;
}

void NativePlatform::onThreadCreated(std::thread::id id, const std::string& name)
{
    (void)id;
    (void)name;
}

void NativePlatform::setLogMessageHandler(LogHandler onLog)
{
    g_logHandler = onLog;
}

NativePlatform::LogHandler NativePlatform::getLogMessageHandler()
{
    return g_logHandler;
}

}
