#include "Qualities.hpp"
#include "media/CodecString.hpp"
#include <algorithm>
#include <iterator>

namespace twitch {
void Qualities::clear()
{
    m_qualities.clear();
    m_autoQualities.clear();
    m_current = Quality();
}

bool Qualities::isSupported(Platform& platform, const Quality& quality) const
{
    using namespace twitch::media;
    CodecString codecString = CodecString::parse(quality.codecs);
    // check the codec string and filter out supported avc profiles
    // check the resolution, bitrate and profile are supported for the platform
    bool isCompatible = true;
    for (const auto& entry : codecString.values) {
        if (entry.first == CodecString::avc1) {
            auto capabilities = platform.getVideoDecoderCapabilities(MediaType::Video_AVC);
            CodecString::AVCParameters avcParameters {};
            avcParameters.profile = static_cast<uint8_t>(capabilities.maxProfile);
            avcParameters.level = static_cast<uint8_t>(capabilities.maxLevel);
            codecString.parseAVCParameters(avcParameters);

            isCompatible = avcParameters.profile <= capabilities.maxProfile
                && avcParameters.level <= capabilities.maxLevel
                && std::max(quality.height, quality.width) <= capabilities.maxWidth
                && std::min(quality.height, quality.width) <= capabilities.maxHeight
                && quality.bitrate <= capabilities.maxBitrate;

        } else if (entry.first == CodecString::vp09) {
            auto capabilities = platform.getVideoDecoderCapabilities(MediaType::Video_VP9);
            CodecString::VP9Parameters vp9Parameters {};
            vp9Parameters.profile = static_cast<uint8_t>(capabilities.maxProfile);
            vp9Parameters.level = static_cast<uint8_t>(capabilities.maxLevel);
            codecString.parseVP9Parameters(vp9Parameters);
            // TODO check bit depth

            isCompatible = vp9Parameters.profile <= capabilities.maxProfile
                && vp9Parameters.level <= capabilities.maxLevel
                && quality.width <= capabilities.maxWidth
                && quality.height <= capabilities.maxHeight
                && quality.bitrate <= capabilities.maxBitrate;
        }
    }

    return isCompatible;
}

Quality Qualities::match(const Quality& quality) const
{
    // Choose a quality by name or group if there's an exact match
    for (const auto& q : m_qualities) {
        bool nameMatch = (!q.name.empty() && q.name == quality.name);
        bool groupMatch = (!q.group.empty() && q.group == quality.group);
        if (nameMatch || groupMatch) {
            return q;
        }
    }

    Quality target;
    // If bitrate not provided, choose default
    if (quality.bitrate == 0 || m_autoQualities.empty()) {
        if (!m_qualities.empty()) {
            target = getDefault();
        }
    } else {
        // Choose bitrate with closest bitrate
        target = match(quality.bitrate);
    }

    return target;
}

Quality Qualities::match(int bitrate) const
{
    Quality target;

    if (!m_autoQualities.empty()) {
        // ensure at least one quality target set before filtering
        target = m_autoQualities.front();
    }

    for (const auto& quality : m_autoQualities) {
        target = quality;

        if (bitrate >= quality.bitrate) {
            break;
        }
    }

    return target;
}

void Qualities::reset(Platform& platform, const std::vector<Quality>& qualities)
{
    m_qualities.clear();
    for (const auto& quality : qualities) {
        if (isSupported(platform, quality)) {
            m_qualities.push_back(quality);
        }
    }

    if (!m_qualities.empty()) {
        // sort by bitrate (preferring the default quality first in the list regardless of bitrate)
        int offset = (m_qualities.front().isDefault && m_qualities.size() > 1) ? 1 : 0;
        std::sort(m_qualities.begin() + offset, m_qualities.end(),
            [](const Quality& a, const Quality& b) { return a.bitrate > b.bitrate; });

        // find default bitrate
        auto it = std::find_if(m_qualities.begin(), m_qualities.end(),
            [](const Quality& quality) { return quality.isDefault; });

        int defaultBitrate = it != m_qualities.end() ? it->bitrate : std::numeric_limits<int>::max();

        // add auto-select qualities
        std::copy_if(m_qualities.begin(), m_qualities.end(), std::back_inserter(m_autoQualities),
            [defaultBitrate](const Quality& quality) {
                return quality.autoSelect && quality.bitrate <= defaultBitrate;
            });
    }
}

void Qualities::remove(const Quality& quality, bool autoOnly)
{
    m_autoQualities.erase(std::remove(m_autoQualities.begin(), m_autoQualities.end(), quality), m_autoQualities.end());
    if (!autoOnly) {
        m_qualities.erase(std::remove(m_qualities.begin(), m_qualities.end(), quality), m_qualities.end());
    }
}

const Quality& Qualities::getDefault() const
{
    auto it = std::find_if(m_qualities.begin(), m_qualities.end(),
        [](const Quality& quality) {
            return quality.isDefault;
        });
    return it != m_qualities.end() ? *it : m_qualities.front();
}

void Qualities::setCurrent(const Quality& quality)
{
    m_current = quality;
}

void Qualities::setSelected(const Quality& quality)
{
    m_previous = m_selected;
    m_selected = quality;
}
}
