#include "ScopedScheduler.hpp"
#include "debug/trace.hpp"
#include <algorithm>

namespace twitch {
ScopedScheduler::ScopedScheduler(std::shared_ptr<Scheduler> scheduler)
    : m_scheduler(std::move(scheduler))
    , m_cancelled(false)
{
}

ScopedScheduler::~ScopedScheduler()
{
    cancel();
}

std::shared_ptr<Cancellable> ScopedScheduler::schedule(Scheduler::Action action, Microseconds time, bool repeating)
{
    std::lock_guard<RecursiveMutex> lock(m_mutex);
    removeExpired();
    if (m_cancelled) {
        return std::make_shared<NoOpTask>();
    } else {
        auto cancellable = m_scheduler->schedule(action, time, repeating);
        m_tasks.emplace_back(cancellable);
        return cancellable;
    }
}

void ScopedScheduler::scheduleAndWait(Scheduler::Action action)
{
    const ScopedScheduler* self = this;
    self->scheduleAndWait(std::move(action));
}

void ScopedScheduler::scheduleAndWait(Scheduler::Action action) const
{
    {
        std::lock_guard<RecursiveMutex> lock(m_mutex);
        if (m_cancelled) {
            return;
        }
    }
    m_scheduler->scheduleAndWait(std::move(action));
}

void ScopedScheduler::cancel()
{
    std::lock_guard<RecursiveMutex> lock(m_mutex);
    if (!m_cancelled) {
        m_cancelled = true;
        for (auto& task : m_tasks) {
            auto cancellable = task.lock();
            if (cancellable) {
                cancellable->cancel();
            }
        }
        m_tasks.clear();
    }
}

void ScopedScheduler::removeExpired()
{
    auto isExpired = [](const std::weak_ptr<Cancellable>& ptr) { return ptr.expired(); };
    m_tasks.erase(std::remove_if(m_tasks.begin(), m_tasks.end(), isExpired), m_tasks.end());
}
}
