#pragma once

#include "playercore/platform/Scheduler.hpp"
#include "util/Concurrency.hpp"
#include <memory>
#include <vector>

namespace twitch {
/**
 * Scheduler wrapper in which all outstanding tasks are automatically cancelled when the object is
 * destroyed.
 *
 * Takes an input Scheduler instance and delegates schedule calls to it, storing weak references
 * to the returned Cancellable objects. When this ScopedScheduler instance is deleted (goes out of
 * scope) all active Cancellable references to the outstanding tasks are cancelled to ensure they
 * don't run.
 */
class ScopedScheduler : public Scheduler, public Cancellable {
public:
    explicit ScopedScheduler(std::shared_ptr<Scheduler> scheduler);
    ~ScopedScheduler() override;
    ScopedScheduler(const ScopedScheduler&) = delete;
    const ScopedScheduler& operator=(const ScopedScheduler&) = delete;

    std::shared_ptr<Cancellable> schedule(Scheduler::Action action, Microseconds time = Microseconds::zero(), bool repeating = false) override;
    void scheduleAndWait(Scheduler::Action action) override;
    void scheduleAndWait(Scheduler::Action action) const;
    std::shared_ptr<Scheduler> share() { return m_scheduler; }
    void cancel() override;

private:
    struct NoOpTask : Cancellable {
        ~NoOpTask() override = default;
        void cancel() override {}
    };

    void removeExpired();

    std::shared_ptr<Scheduler> m_scheduler;
    std::vector<std::weak_ptr<Cancellable>> m_tasks;
    mutable RecursiveMutex m_mutex;
    bool m_cancelled;
};
}
